<?php

/*
Plugin Name: WP ParaMedic
Plugin URI: http://wpmedic.tech/paramedic/
Description: Automated Health Check for All Plugin and Theme Errors
Version: 0.9.6
Author: Tony Hayes
Author URI: http://wpmedic.tech
*/

/* --- START DOCS ---

# WP Paramedic

## The Black Screen of Life

### Brought to you by WP Medic
Love this tool? [Become a WP Medic Patron](https://patreon.com/wpmedic)

[WP Paramedic Home](http://wpmedic.tech/paramedic/) - [GitHub](https://github.com/majick777/paramedic/)

***

### Introduction



#### Security Warning

Even though the security risk is very low (revealing installed plugin and theme versions)
- it is still preferable that you do not leave Paramedic installed during normal operation
of a live site, but rather drop it in, use it, and then remove it when you are done.

Practically speaking however, if your site (or admin) is crashed, in an emergency there
are situations where you may need to keep using Paramedic to narrow down the problem
further. In this case you will want to setup a simple security key to access Paramedic.
(see Security Key Setup in the Installation section)


### Installation

1. Simply copy this one file in your WordPress install `/wp-content/mu-plugins/` directory.
2. If this directory does not exist then you will need to create it first.
  * remembering to `chown owner:group mu-plugins` to match your existing installation.
3. Note because of the `000-` filename prefix, this file will load before other must-use plugins.
  * this is intentional so that it can test and track the results of other must-use plugins.
4. Go to your site and append ?paramedic to the URL to load the ParaMedic test interface.
  * eg. `http://yoursite.com/?paramedic`

#### Security Key Setup

If you are leaving Paramedic installed for any length of time, it is recommended you
require an authentication key to access the interface. Set this constant in `wp-config.php`:

```
define('PARAMEDIC_MUSTAUTH', true);
```

And define a hard-coded password-style authentication string there as well:

```
define('PARAMEDIC_KEY', 'myaccesskey');
```

Once done, to access Paramedic you will need to provide the key in the URL:

`http://yoursite.com/?paramedic=myaccesskey`
or `http://yoursite.com/?paramedic&key=myaccesskey`


### Usage Notes

The Paramedic Interface is available by appending ?paramedic to your site URL:
eg. `http://yoursite.com/?paramedic`

Alternatively you can use a direct Querystring to conduct test (see list below)
eg. `http://yoursite.com/?paramedic&plugintest=all`

#### Trigger Test Querystrings

| ?paramedic			| launch ParaMedic tests and/or display test launch form menu |
| &coretest=			| active/yes/1 (automatically triggered for any other tests) |
| &themetest=			| active/yes/1, inactive, network, all	{comma-separated theme slugs} |
| &plugintest=			| active/yes/1, inactive, network, all	{comma-separated plugin slugs} |

| &testall=				| yes or 1	| test all plugins and themes |
|						| active 	| test all active plugins and active theme |
|						| inactive 	| tests all inactive plugins and themes |
|						| network 	| tests network active plugins and active blog themes |

Note: combinations can trigger multiple tests in series.

#### Test Specific Resources

*this documentation may need updating*
| &testplugins=			| comma separated string of plugins to test |
| &testthemes=			| comma separated string of themes to test |

#### Test Querystrings

| Querystring 	| Possible Values 			| Description |
| &testid=		| id						| arbitrary ID (for identifying test result log) |
| &testurl=		| {FULL URL}				| eg. http://example.com/ (defaults to site_url('/')) |
| &usetheme=	| themeslug					| forces the use of a specified theme with plugin tests |
| &silent=		| yes or 1					| test results to log file only, no screen output |
| &user=		| current/guest/userid		| user to be used during tests (default: current user) |
| &userid=		| numeric					| [*KEY REQUIRED*] set specific user ID (if user=userid) |
| &method=		| wp/curl/multicurl			| set test method for Curl (default wp) |
| &timelimit=	| numeric					| maximum time limit for all tests to complete in |
| &testlog=		| yes or 1					| whether to log test results to instance file |
| &logpath=		| directory path			| directory path for logging and test results files |
| &cleanup=		| yes or 1					| cleanup test instance log files (can be used with logpath) |
| &inverse=		| yes/1, no/0 or all		| do inverse plugin test on plugin error (default yes) |
| &repair=		| yes or 1					| [*KEY REQUIRED*] deactivate plugins with fatal errors |
| &debug		| yes/echo, log or logecho	| output tester debugger information on the page |

Note: security key constant (see below) is required for setting `userid` or `repair`.

#### Test Constants

Set these in your `wp-config.php`.

##### Security Constants

| PARAMEDIC_KEY			| set/override access key for tests (*required for specific user ID tests*) |
| PARAMEDIC_MUSTAUTH	| set to true to *require* the access key for any tests and test menu display |

##### Settings Constants

Setting of these constants *override* any querystrings values (see Test Querystrings section.)
If set, remember that results from querystring-triggered tests may not be what you expected.

| PARAMEDIC_ID			| override for test instance ID |
| PARAMEDIC_TOKEN		| override for test internal token |
| PARAMEDIC_URL			| override for test request URL |
| PARAMEDIC_TIME		| override for test maximum execution time |
| PARAMEDIC_USER		| override for test user (current, guest, user ID, or username) |
| PARAMEDIC_METHOD		| override for test method (wp, curl or multicurl) |
| PARAMEDIC_TESTLOG		| override for test result logging switch |
| PARAMEDIC_LOGPATH		| override for debug log path (default /wp-content/debug.log) |
| PARAMEDIC_SILENT		| override for silent test output (ie. logging only) |
| PARAMEDIC_THEME_DIR	| override for theme root directory |
| PARAMEDIC_PLUGIN_DIR  | override for plugin root directory |
| PARAMEDIC_REPAIR		| override for auto-deactivate plugins causing fatal errors |
| PARAMEDIC_DEBUG		| override for displaying all debug information on test page |


#### Must-Use Plugin Tests

*This feature is still under consideration and development.*

As this file is itself a must-use plugin, in order to test these you will need to add
this snippet to your `wp-config.php` to make testing these possible:

```
if (isset($_REQUEST['paramedic'])) {
	define('MUPLUGINDIR', ABSPATH.'/wp-content/mu-plugins/paramedic/');
	define('WPMU_PLUGIN_DIR', ABSPATH.'/wp-content/mu-plugins/paramedic/');
	define('WPMU_PLUGIN_URL', $_SERVER['HTTP_HOST'].'/wp-content/mu-plugins/paramedic/');
}
```

1. Add the above code snippet to your installation `wp-config.php` file.
2. If needed, adjust the paths and URLs to match your WordPress installation.
3. Then copy 000-paramedic.php to `/wp-content/mu-plugins/paramedic/`
4. Copy any must-use plugins that you want to test into the same directory.
5. Remember to check and maybe change the file owner permissions to match your intall.
6. Run WP Paramedic tests as normal and it will be able to test the `mu-plugins`

--- END DOCS --- */

// === Changelog ===

// = 0.9.6 = 
// - added sanitization to request variables
// - removed logpath setting via querystring request

// = 0.9.5 =
// - fix to random undefined index warnings
// - add wp_remote timeout and args filters
// - add resource usage statistic tracking
// - re-enable multi-curl test request method


// Development TODOs
// -----------------
// * custom plugin / theme test multi-selectors
// - test theme parent / child selection options
// * do inverse plugin test on single plugin test failure
// + add test debug log cleanup command
// + add self destruct command
// ? add test method for mu-plugins (via WPMU_PLUGIN_DIR, WPMU_PLUGIN_URL, MUPLUGINDIR)
// -- ref: https://www.binarytemplar.com/plugins/disablemu/
// - recheck Error Handler Return Codes
// -- ref: https://stackoverflow.com/questions/51963597/suppress-php-filemtime-errors-when-a-file-does-not-exist

// ========================
// === Tester Structure ===
// ========================
// - User Helper Functions
// - Tester Setup
// - Tester Interface
// - Test Method Handlers
// - Error Handler Class
// - Test Result Collater
// - Output Test Results
// - Helper Functions
// - Handle Test Requests
// - Check Test Triggers
// - Start Tests
// ========================

// --- no direct access ---
if ( !defined( 'ABSPATH' ) ) {
	exit;
}


// -----------------------------
// === User Helper Functions ===
// -----------------------------

// --------------------------
// Recheck User Authorization
// --------------------------
if ( !function_exists( 'paramedic_recheck_user_auth' ) ) {
	function paramedic_recheck_user_auth() {
		global $paramedic;
		// --- mostly may need to load pluggable early ---
		if ( !function_exists( 'is_user_logged_in' ) ) {
			include_once ABSPATH . WPINC . '/pluggable.php';
		}
		// 0.9.5: load default constants
		if ( !defined( 'COOKIEHASH' ) ) {
			if ( !function_exists( 'wp_cookie_constants' ) ) {
				include_once ABSPATH . WPINC . '/default-constants.php';
			}
			wp_cookie_constants();
		}
		if ( is_user_logged_in() && current_user_can( 'manage_options' ) ) {
			$paramedic['authed'] = true;
		}
	}
}

// -------------------------------------
// Adjust Curl Timeout for wp_remote API
// -------------------------------------
if ( !function_exists( 'paramedic_wp_adjust_timeout' ) ) {
	function paramedic_wp_adjust_timeout( $handle ) {
		curl_setopt( $handle, CURLOPT_CONNECTTIMEOUT, 10 );
		curl_setopt( $handle, CURLOPT_TIMEOUT, 20 );
	}
}

// ----------------
// Load Stats Start
// ----------------
if ( !function_exists( 'paramedic_stats_start' ) ) {
	function paramedic_stats_start( $resource ) {
		global $paramedic;
		$resource = str_replace( array( '/', '\\' ), '--', $resource );
		$paramedic['loadtimes'][$resource]['start'] = microtime( true );
		$paramedic['loadmemory'][$resource]['start'] = memory_get_usage();
		$paramedic['loadqueries'][$resource]['start'] = get_num_queries();
	}
}

// --------------
// Load Stats End
// --------------
if ( !function_exists( 'paramedic_stats_end' ) ) {
	function paramedic_stats_end( $resource ) {
		global $paramedic;
		$resource = str_replace( array( '/', '\\' ), '--', $resource );

		// --- memory usage ---
		$paramedic['loadtimes'][$resource]['end'] = microtime( true );
		$difference = $paramedic['loadtimes'][$resource]['end'] - $paramedic['loadtimes'][$resource]['start'];
		$paramedic['loadtimes'][$resource]['time'] = $difference;

		// --- load time ---
		$paramedic['loadmemory'][$resource]['end'] = memory_get_usage();
		$difference = $paramedic['loadmemory'][$resource]['end'] - $paramedic['loadmemory'][$resource]['start'];
		$paramedic['loadmemory'][$resource]['memory'] = $difference;

		// --- load queries ---
		$paramedic['loadqueries'][$resource]['end'] = get_num_queries();
		$difference = $paramedic['loadqueries'][$resource]['end'] - $paramedic['loadqueries'][$resource]['start'];
		$paramedic['loadqueries'][$resource]['queries'] = $difference;
	}
}

// -------------
// Debug Message
// -------------
if ( !function_exists( 'paramedic_debug' ) ) {
	function paramedic_debug( $message ) {
		global $paramedic;
		if ( !$paramedic['debug'] ) {return;}

		// --- maybe debug to log file ---
		if ( $paramedic['testlog'] ) {
			if ( in_array( $paramedic['debug'], array( 'log', 'logecho' ) ) ) {
				$logpath = dirname( $paramedic['logpath'] ) . '/paramedic-test-' . $paramedic['testid'] . '.log';
				error_log( $message, 3, $logpath );
				// echo "***".$logpath."***";
			}
		}

		// --- maybe output debug message ---
		if ( $paramedic['silent'] ) {return;}

		if ( in_array( $paramedic['debug'], array( 'echo', 'logecho' ) ) ) {
			$message = str_replace( PHP_EOL, '<br>', $message );
			echo $message;
		}
	}
}

// --------------------
// === Tester Setup ===
// --------------------

// --- declare tester global ---
global $paramedic;
$paramedic['version'] = '0.9.6';

// --- set or generate a new test ID ---
if ( defined( 'PARAMEDIC_ID' ) ) {
	$testid = PARAMEDIC_ID;
} elseif ( isset( $_REQUEST['testid'] ) ) {
	$checktestid = trim( sanitize_text_field( $_REQUEST['testid'] ) );
	$checktestid = preg_replace( "/[^0-9a-zA-Z]/i", '', $checktestid );
	if ( '' != $checktestid ) {
		$testid = $checktestid;
	}
}
if ( !isset( $testid ) ) {
	// --- new test so make sure we have a new matching transient ---
	$testid = get_option( 'paramedic_test_id' );
	if ( !$testid ) {
		$testid = '00001';
	}
	// $testid = rand( 10000, 99999 );
	if ( get_transient( 'paramedic_token_' . $testid ) ) {
		while ( get_transient( 'paramedic_token_' . $testid ) ) {
			// $testid = rand( 10000, 99999 );
			$testid = (int) $testid + 1;
			$testid = (string) $testid;
		}
	}
}
$paramedic['testid'] = $testid;

// --- get active plugins and theme now (pre-filtering) ---
$paramedic['plugins']['active'] = get_option( 'active_plugins' );
$paramedic['themes']['active'] = get_option( 'stylesheet' );
$paramedic['themes']['parent'] = get_option( 'template' );

// --- maybe get multisite network activated plugins (pre-filtering) ---
if ( is_multisite() ) {

	$paramedic['networkplugins'] = get_site_option( 'active_sitewide_plugins' );

	// TODO: use wp_get_active_network_plugins to validate network plugins
	// note: uses array_keys from active_sitewide_plugins
	// $network_plugins = wp_get_active_network_plugins();
	// foreach ( $network_plugins as $i => $plugin ) {
	//	$network_plugins[$i] = str_replace( WP_PLUGIN_DIR, '', $plugin );
	// }
}

// --- set Debug Log Path ---
// 0.9.3: added missing check for constant override
// 0.9.4: fix to use file path instead of directory path
if ( defined( 'WP_CONTENT_DIR' ) ) {
	$paramedic['logpath'] = WP_CONTENT_DIR . '/debug.log';
} else {
	$paramedic['logpath'] = ABSPATH . 'wp-content/debug.log';
}
// 0.9.6: check directory exists instead of file exists
if ( defined( 'PARAMEDIC_LOGPATH' ) && is_dir( dirname( PARAMEDIC_LOGPATH ) ) ) {
	$paramedic['logpath'] = PARAMEDIC_LOGPATH;
}
// 0.9.6: disabled this option (security risk)
// elseif ( isset( $_REQUEST['logpath'] ) && file_exists( $_REQUEST['logpath'] ) ) {
//	$paramedic['logpath'] = $_REQUEST['logpath'];
// }

// --- maybe set Plugin Test Directory Override ---
// 0.9.2: added for testing with alternative plugin root directory
// 0.9.3: added constant override for root plugin directory
if ( defined( 'PARAMEDIC_PLUGIN_DIR' ) ) {
	$plugindir = PARAMEDIC_PLUGIN_DIR;
	if ( !is_dir( $plugindir ) ) {
		$plugindir = WP_CONTENT_DIR . '/' . $plugindir . '/';
	}
} elseif ( isset( $_REQUEST['plugindir'] ) ) {
	$plugindir = WP_CONTENT_DIR . '/' . $_REQUEST['plugindir'] . '/';
	// 0.9.3: security: disallow paths above /wp-content/ in querystring overrides
	if ( strpos( realpath( $plugindir ), WP_CONTENT_DIR ) !== 0 ) {
		unset( $plugindir );
	}
}
if ( !isset( $plugindir ) || !is_dir( $plugindir ) ) {
	$plugindir = trailingslashit( WP_PLUGIN_DIR );
}
$paramedic['plugindir'] = $plugindir;

// --- maybe set Theme Test Directory Override ---
// 0.9.3: added theme root directory overrides
if ( defined( 'PARAMEDIC_THEME_DIR' ) ) {
	$themedir = PARAMEDIC_THEME_DIR;
	if ( !is_dir( $themedir ) ) {
		$themedir = WP_CONTENT_DIR . '/' . $themedir . '/';
	}
} elseif ( isset( $_REQUEST['themedir'] ) ) {
	$themedir = WP_CONTENT_DIR . '/' . $_REQUEST['themedir'] . '/';
	// 0.9.3: security: disallow paths above /wp-content/ in querystring overrides
	if ( strpos( realpath( $themedir ), WP_CONTENT_DIR ) !== 0 ) {
		unset( $themedir );
	}
}
if ( !isset( $themedir ) || !is_dir( $themedir ) ) {
	$themedir = WP_CONTENT_DIR . '/themes/';
}
$paramedic['themedir'] = $themedir;

// --- maybe set debug mode ---
// 0.9.4: added tester debug output mode with echo/log/logecho value options
$paramedic['debug'] = false;
if ( defined( 'PARAMEDIC_DEBUG' ) ) {
	$debugoutput = PARAMEDIC_DEBUG;
} elseif ( isset( $_REQUEST['debug'] ) ) {
	$debugoutput = sanitize_text_field( $_REQUEST['debug'] );
}
if ( isset( $debugoutput ) && in_array( $debugoutput, array( 'yes', '1', 'echo' ) ) ) {
	$paramedic['debug'] = 'echo';
} elseif ( isset( $debugoutput ) && in_array( $debugoutput, array( 'log', 'logecho' ) ) ) {
	$paramedic['debug'] = $debugoutput;
}

// --- check for auth key ---
// note: user authentication is checked separately
// 0.9.4: added security key and user capability check
$paramedic['authed'] = false;
if ( defined( 'PARAMEDIC_KEY' ) && isset( $_REQUEST['key'] ) ) {
	if ( sanitize_text_field( $_REQUEST['key'] ) == PARAMEDIC_KEY ) {
		$paramedic['authed'] = true;
	}
}


// ------------------------
// === Tester Interface ===
// ------------------------

// -------------------
// ParaMedic Logo Icon
// -------------------
if ( !function_exists( 'paramedic_logo_icon' ) ) {
	function paramedic_logo_icon() {
		echo '<img src="data:image/png;base64,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" alt="ParaMedic" border="0"/>';
	}
}

// -------------
// WP Medic Icon
// -------------
if ( !function_exists( 'paramedic_medic_icon' ) ) {
	function paramedic_medic_icon() {
		echo '<img src="data:image/png;base64,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" alt="WP Medic" border="0" />';
	}
}

// --------------
// Patreon Button
// --------------
// 0.9.4: added patreon button image
if ( !function_exists( 'paramedic_patreon_button' ) ) {
	function paramedic_patreon_button() {
		echo '<img id="patreon-button-image" src="data:image/png;base64,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" alt="Become a Patron" border="0" />';
	}
}

// ---------------------
// Output Interface Form
// ---------------------
// 0.9.4: added form interface
// TODO: maybe add translation wrappers?
if (!function_exists('paramedic_interface')) {
	function paramedic_interface() {

		global $paramedic;

		if ( $paramedic['silent'] || $paramedic['interface'] ) {
			return;
		}

		echo '<html><body><div id="header">' . "\n";

			// --- header styles ---
			// 0.9.4: added patreon button opacity and change on hover
			// 0.9.4: inverted colour scheme to white on black
			echo "<style>body, input[type='text'], select {background:#000; color:#FFF;}
			input[type='text'] {border-width: 1px; padding: 2px 5px;}
			input[type='radio'], input[type='checkbox'] {}
			.help-text {font-size:9pt;} .helper {font-size: 12px; color:#0000BB; margin-left:10px;}
			#header-table, #menu, .input-table {font-family: arial,helvetica; font-size: 14px;}
			#heading {font-size: 30px; margin: 5px;} #version {font-size:18px;}
	        #patron-button, #test-submit {background: #00DDFF; border-radius: 5px; padding: 5px 10px;
	            border-color: #00BBFF; color: #FFF; text-shadow: 1px 1px #888; margin-top:5px;}
	        #patron-button:hover, #test-submit:hover {background: #00EEFF;}
			#patreon-button-image {opacity: 0.9;} #patreon-button-image:hover {opacity: 1;}
			#check-button {background: #444; border-radius:5px; color: #FFF;
				text-shadow: 1px 1px #888; border-width: 1px;}
	        #check-button:hover {background: #777;}
	        #icon-parent {position: relative; width: 96px; height: 96px;}
	        #icon-child {position: absolute; margin-left:10000px; display:none;}
			.link {text-decoration: none; line-height: 22px; color: #00DDDD;}
			.link:hover {text-decoration: underline; color: #00FFFF;}
			.link:visited {color: #00AAAA;}
	        .input-table, #advanced-options {margin-top:10px; margin-bottom:10px;}
			#user, #themetest, #plugintest, #testall {margin-top:5px; margin-bottom:5px;}
			.result-row {background:#333;}
	        </style>" . "\n";

			// --- ParaMedic display header ---
			echo '<center><table id="header-table"><tr><td align="center">' . "\n";

				// --- ParaMedic icon display ---
				echo '<div id="icon-parent"><div id="icon-child">' . "\n";
					echo '<a href="https://wpmedic.tech/paramedic/" target=_blank">' . "\n";
					paramedic_logo_icon();
				echo '</a></div></div>' . "\n";

			echo '</td><td width="50"></td><td align="center">' . "\n";

				// --- heading display ---
				echo '<h2 id="heading">' . "\n";
				$heading = __( 'WP ParaMedic', 'paramedic' );
				while ( '' != $heading ) {
					if ( !isset( $color ) || ( $color == "#ED1B24" ) ) {
						$color = "#357BB9";
					} else {
						$color = "#ED1B24";
					}
					$char = substr( $heading, 0, 1 );
					if ( ' ' == $char ) {
						$color = "#ED1B24";
					}
					echo '<span style="color:' . esc_attr( $color ) . ';">' . esc_html( $char ) . '</span>';
					$heading = substr( $heading, 1, strlen( $heading ) );
				}
				echo '</h2>' . "\n";

				echo '<table><tr><td>' . "\n";
					// --- version display ---
					echo '<span id="version">v' . esc_html( $paramedic['version'] ) . '</span>';
				echo '</td><td width="20"></td><td>';
					// --- update check button ---
					$title = __( 'Check for Updates', 'paramedic' );
					echo '<input type="button" id="check-button" onclick="paramedic_check_updates();" value="' . esc_attr( __( 'Check', 'paramedic') ) . '" title="' . esc_attr( $title ) . '">' . "\n";
				echo '</td></tr></table>' . "\n";

			echo '</td><td width="50"></td><td align="center">';

				// --- WP Medic icon display ---
				echo '<a href="https://wpmedic.tech/" target="_blank">' . "\n";
					paramedic_medic_icon();
				echo '</a><br>' . "\n";

			echo '</td><td width="10"></td><td align="center">';

				// --- credits and patronage button ---
				echo esc_html( __( 'Brought to you by', 'paramedic' ) ) . ' ';
				echo '<a href="https://wpmedic.tech/" class="link" target="_blank">WP Medic</a><br><br>' . "\n";
				echo '<a href="https://patreon.com/wpmedic/" target="_blank">' . "\n";
					// 0.9.4: use patreon button image
					// echo "<input type='button' id='patron-button' value='Become a WP Medic Patron'>
					paramedic_patreon_button();
				echo '</a><br>' . "\n";

			echo '</td><td width="50"></td><td align="center">';

				// --- help links ---
				// TODO: querystring help link (popup?)
				// echo '<a href="?paramedic=help" class="link" target="_blank">' . esc_html( __( 'Help', 'paramedic' ) ) . '</a><br>';
				echo '<a href="https://wpmedic.tech/paramedic/" class="link" target="_blank">' . esc_html( __( 'Docs', 'paramedic' ) ) . '</a><br>' . "\n";
				echo '<a href="https://wordquest.org/support/wp-paramedic/" class="link" target="_blank">' . esc_html( __( 'Support', 'paramedic' ) ) . '</a><br>' . "\n";
				echo '<a href="https://github.com/majick777/wp-paramedic/" class="link" target="_blank">' . esc_html( __( 'Development', 'paramedic' ) ) . '</a><br>' . "\n";

			echo '</td></tr></table></center><br>' . "\n";

			// --- page scripts ---
			// TODO: self-load update check via wp_remote_get
			// TODO: find user helper function
			echo "<script>
			/* check for updates */
			function paramedic_check_updates() {
				version = '" . esc_js( $paramedic['version'] ) . "';
				url = 'https://wpmedic.tech/?paramedic-version-check='+version;
				document.getElementById('updates-frame').src = url;
			}
			/* show advanced options */
			function paramedic_show_advanced() {
				document.getElementById('show-advanced').style.display = 'none';
				document.getElementById('hide-advanced').style.display = '';
				document.getElementById('advanced-options').style.display = '';
			}
			/* hide advanced options */
			function paramedic_hide_advanced() {
				document.getElementById('hide-advanced').style.display = 'none';
				document.getElementById('show-advanced').style.display = '';
				document.getElementById('advanced-options').style.display = 'none';
			}
			/* check for user ID selection */
			function paramedic_check_user() {
				select = document.getElementById('user');
				value = select.options[select.selectedIndex].value;
				if (value == 'userid') {document.getElementById('userid-wrap').style.display = '';}
				else {document.getElementById('userid-wrap').style.display = 'none';}
			}
			/* check for custom theme selection */
			function paramedic_check_themes() {
				select = document.getElementById('themetest');
				value = select.options[select.selectedIndex].value;
				if (value == 'custom') {document.getElementById('theme-wrap').style.display = '';}
				else {document.getElementById('theme-wrap').style.display = 'none';}
			}
			/* check for custom plugin selection */
			function paramedic_check_plugins() {
				select = document.getElementById('plugintest');
				value = select.options[select.selectedIndex].value;
				if (value == 'custom') {document.getElementById('plugin-wrap').style.display = '';}
				else {document.getElementById('plugin-wrap').style.display = 'none';}
			}
			/* find user helper function */
			function paramedic_find_user() {}

			/* logo animation fun! */
			setTimeout(function() {
				var accelerate = 15;
				var offset = -((window.innerWidth - document.getElementById('header-table').clientWidth) / 2) - 96;
				document.getElementById('icon-child').style.display = 'block';
				document.getElementById('icon-child').style.marginLeft = offset+'px';
				var animation = setInterval(function() {
					offset = offset+accelerate;
					if (offset > -128) {accelerate = 10;}
					if (offset > -96) {accelerate = 5;}
					if (offset > 0) {offset = 0; clearInterval(animation);}
					document.getElementById('icon-child').style.marginLeft = offset+'px';
				}, 50);
			}, 1000);
			</script>";

			// --- check updates iframe ---
			echo '<iframe src="javascript:void(0)" style="display:none;" name="updates-frame" id="updates-frame" width="400" height="50"></iframe>' . PHP_EOL;

		echo '</div>' . "\n";

		// --- test menu ---
		echo '<div id="menu"><center>';

			// --- security key message ---
			paramedic_recheck_user_auth();

			// --- key required for user specific user message ---
			if ( !$paramedic['authed'] ) {
				// TODO: add better security key usage info here ?
				// echo "<a href='javascript:void(0);' onclick='keymessage();'>What's this?</a>
				// echo "<br><br>";
			}

			// --- start form ---
			echo '<form action="">' . "\n";

				echo '<input type="hidden" name="paramedic" value="">' . "\n";

				// --- input table ---
				echo '<table class="input-table"><tr>' . "\n";

					// --- check advanced option values ---
					// - silent -
					$advanced = false;
					if ( defined( 'PARAMEDIC_SILENT' ) ) {
						$silent = PARAMEDIC_SILENT; $advanced = true;
					} elseif ( isset( $_REQUEST['silent'] ) ) {
						$silent = $_REQUEST['silent']; $advanced = true;
					} else {
						$silent = '';	
					}
					// - method -
					if ( defined( 'PARAMEDIC_METHOD' ) ) {
						$method = PARAMEDIC_METHOD; $advanced = true;
					} elseif ( isset( $_REQUEST['method'] ) ) {
						$method = sanitize_text_field( $_REQUEST['method'] ); $advanced = true;
					} else {
						$method = 'wp';
					}
					// - test logging
					if ( defined( 'PARAMEDIC_TESTLOG' ) ) {
						$testlog = PARAMEDIC_TESTLOG; $advanced = true;
					} elseif ( isset( $_REQUEST['testlog'] ) ) {
						$testlog = sanitize_text_field( $_REQUEST['testlog'] ); $advanced = true;
					} else {
						$testlog = 'yes';
					}
					// - log path -
					if ( defined( 'PARAMEDIC_LOGPATH' ) ) {
						$logpath = PARAMEDIC_LOGPATH; $advanced = true;
					} elseif ( isset( $_REQUEST['logpath'] ) ) {
						$logpath = sanitize_text_field( $_REQUEST['logpath'] ); $advanced = true;
					} else {
						$logpath = '';
					}
					// - debug -
					if ( defined( 'PARAMEDIC_DEBUG' ) ) {
						$debug = PARAMEDIC_DEBUG; $advanced = true;
					} elseif ( isset( $_REQUEST['debug'] ) ) {
						$debug = sanitize_text_field( $_REQUEST['debug'] ); $advanced = true;
					} else {
						$debug = false;
					}

					// --- test URL ---
					echo '<td><b>' . esc_html( __( 'Test URL', 'paramedic' ) ) . '</b>:</td>' . "\n";

					echo '<td width="10"></td><td>' . "\n";
	                    // --- set base site test URL ---
	                    // 0.9.5: check both constant and current request
						$testurl = '';
						$readonly = false;
	                    if ( defined( 'PARAMEDIC_URL' ) ) {
							$testurl = PARAMEDIC_URL;
							$readonly = true;
						} elseif ( isset( $_REQUEST['testurl'] ) ) {
							$testurl = trim($_REQUEST['testurl']);
						}
	                    $placeholder = site_url( '/' ) . ' (' . esc_attr( __( 'default', 'paramedic' ) ) . ')'; // default to site home
	                    echo '<input type="text" name="testurl" value="' . esc_url_raw( $testurl ) . '"';
						if ( $readonly ) {
							echo ' readonly="readonly"';
						}
						echo ' style="width:300px;" placeholder="' . esc_attr( $placeholder ) . '">' . "\n";
					echo '</td><td width="20"></td>' . "\n";

					// --- test ID ---
					$readonly = false;
					echo '<td><b>' . esc_html( __( 'Test ID', 'paramedic' ) ) . '</b>:</td>' . "\n";
					echo '<td width="10"></td><td>' . "\n";
						if ( defined( 'PARAMEDIC_ID') ) {
							$testid = $paramedic['testid'];
							$readonly = true;
						} else {
							// 0.9.5: auto-increment test ID to prevent test conflicts
							$testid = $paramedic['testid'] + 1;
						}
						echo '<input type="text" name="testid" value="' . esc_attr( $paramedic['testid'] ) . '"';
						if ( $readonly ) {
							echo ' readonly="readonly"';
						}
						echo ' style="width:50px; text-align:center;">';
					echo '</td><td width="20"></td>' . "\n";

					// --- core test (required) ---
					echo '<td><b>' . esc_html( __( 'Core Test', 'paramedic' ) ) . '</b>:</td>' . "\n";
					echo '<td width="10"></td><td>' . "\n";
						$title = __( 'Core Test is always required to compare with later tests.', 'paramedic' );
						echo '<input type="checkbox" name="coretest" value="yes" title="' . esc_attr( $title ) . '" checked="checked" disabled="disabled"></td>' . "\n";
						echo '<input type="hidden" name="coretest" value="yes">' .  "\n";
					echo '<td width="20"></td>' . "\n";

					// --- must use plugins test ---
					// TEMP: currently always required (future option)
					echo '<td><b>' . __( 'Must Use Plugins', 'paramedic' ) . '</b></td>' . "\n";
					echo '<td width="10"></td>' . "\n";
					echo '<td>' . "\n";
						// if (isset($_REQUEST['testmuplugins']) && ($_REQUEST['testmuplugins'] == 'yes')) {$checked = ' checked';} else {$checked = '';}
						// echo "<input type='checkbox' name='testmuplugins' value='yes'".$checked.">";
						// TEMP: show always enabled checkbox
						$title = __( 'Must use plugins test is currently included in core test.', 'paramedic' );
						echo '<input type="checkbox" name="testmuplugins" value="yes" title="' . esc_attr( $title ) . '" checked="checked" disabled="disabled">' . "\n";
						echo '<input type="hidden" name="testmuplugins" value="yes">' . "\n";
					echo '</td><td width="20"></td>' . "\n";

					echo '<td>' . "\n";
						if ( $advanced ) {$show = ' style="display:none;"'; $hide = '';}
						else {$show = ''; $hide = ' style="display:none;"';}
						echo '<div id="show-advanced"';
						if ( $advanced ) {
							echo ' style="display:none;"';
						}
						echo '>' . "\n";
							echo '<a href="javascript:void(0);" class="link" onclick="paramedic_show_advanced();">[+] ' . esc_html( __( 'Advanced Options', 'paramedic' ) ) . '</a>' . "\n";
						echo '</div>' . "\n";
						echo '<div id="hide-advanced"';
						if ( !$advanced ) {
							echo ' style="display:none;"';
						}
						echo '>' . "\n";
							echo '<a href="javascript:void(0);" class="link" onclick="paramedic_hide_advanced();">[-] ' . esc_html( __( 'Advanced Options', 'paramedic' ) ) . '</a>' . "\n";
						echo '</div>' . "\n";
					echo '</td>' . "\n";

				echo '</tr></table>' . "\n";

				// --- advanced options ---
				// (note: also check defines for these options)
				echo '<div id="advanced-options"';
				if ( !$advanced ) {
					echo ' style="display:none;"';
				}
				echo '>' . "\n";
					echo '<table class="input-table"><tr>' . "\n";

						// --- method ---
						echo '<td>' . esc_html( __( 'Method','paramedic' ) ) . ':</td><td>' . "\n";
							if ( defined( 'PARAMEDIC_METHOD' ) ) {
								echo '<input type="hidden" name="method" value="' . esc_attr( $method ) . '">' . PHP_EOL;
							}
							if ( !isset( $method ) || ( $method == 'wp' ) ) {
								$checked = ' checked="checked"';
							} else {
								$checked = '';
							}
							echo '<input type="radio" name="method" value="wp"';
							if ( !isset( $method ) || ( $method == 'wp' ) ) {
								echo ' checked="checked"';
							}
							if ( defined( 'PARAMEDIC_METHOD' ) ) {
								echo ' disabled="disabled"';
							}
							echo '> ' . esc_html( __( 'WP', 'paramedic' ) ) . ' ' . "\n";
	                        echo ' <input type="radio" name="method" value="curl"';
							if ( isset( $method ) && ( 'curl' == $method ) ) {
								echo ' checked="checked"';
							}
							if ( defined( 'PARAMEDIC_METHOD' ) ) {
								echo ' disabled="disabled"';
							}
							echo '> ' . esc_html( __( 'Curl', 'paramedic' ) ) . ' ' . "\n";
	                        // 0.9.5: added multicurl option
							echo ' <input type="radio" name="method" value="multicurl"';
							if ( isset( $method ) && ( 'multicurl' == $method ) ) {
								echo ' checked="checked"';
							}
							if ( defined( 'PARAMEDIC_METHOD' ) ) {
								echo ' disabled="disabled"';
							}
							echo '> ' . esc_html( __( 'MultiCurl', 'paramedic' ) ) . ' ' . "\n";
						echo '</td><td width="30"></td><td>' . "\n";

						// --- silent ---
						echo '<td>' . __( 'Silent', 'paramedic' ) . '?</b></td><td>' . "\n";
						if ( defined('PARAMEDIC_SILENT' ) ) {
							echo '<input type="hidden" name="silent" value="' . esc_attr( $silent ) . '">';
						}
						echo '<input type="checkbox" name="silent" value="yes"';
						if ( ( defined( 'PARAMEDIC_SILENT' ) && PARAMEDIC_SILENT ) || ( 'yes' == $silent ) ) {
							echo ' checked="checked"';
						}
						if ( defined( 'PARAMEDIC_SILENT' ) ) {
							echo ' disabled="disabled"';
						}
						echo '>' . "\n";
						echo '</td><td width="30"></td><td>' . "\n";

						// --- test log ---
						echo '<td>' . esc_html( __( 'Create Test Report', 'paramedic' ) ) . '?</b></td><td>' . "\n";
							if ( defined( 'PARAMEDIC_LOG' ) ) {
								 echo '<input type="hidden" name="testlog" value="' . esc_attr( $testlog ) . '">' . "\n";
							}
							echo '<input type="checkbox" name="testlog" value="yes"';
							if ( ( defined( 'PARAMEDIC_LOG' ) && PARAMEDIC_LOG ) || ( $testlog ) ) {
								echo ' checked="checked"';
							}
							if ( defined( 'PARAMEDIC_LOG' ) ) {
								echo ' disabled="disabled"';
							}
							echo '>' . "\n";

						echo '</td><td width="30"></td><td>' . "\n";

						// --- log path ---
						// echo "<td>" . __( 'Log Path', 'paramedic') . ":</b></td><td>";
						//	if ( !defined( 'PARAMEDIC_LOGPATH' ) ) {$readonly = '';} else {$readonly = ' readonly';}
						//	echo "<input type='text' name='logpath' value='" . $logpath . "'" . $readonly . ">";
						// echo "</td><td width='30'></td><td>";

						// --- debug output ---
						echo '<td>' . esc_html( __( 'Debug', 'paramedic' ) ) . '?</b></td><td>' . "\n";
							if ( defined( 'PARAMEDIC_DEBUG' ) ) {
								echo '<input type="hidden" name="debug" value="' . esc_attr( $debug ) . '">';
							}
							echo '<input type="checkbox" name="debug" value="yes"';
							if ( ( defined( 'PARAMEDIC_DEBUG' ) && PARAMEDIC_DEBUG ) || $debug ) {
								echo ' checked="checked"';
							}
							if ( defined( 'PARAMEDIC_DEBUG' ) ) {
								echo ' disabled="disabled"';
							}
							echo '>' . "\n";
						echo '</td>';

					echo '</tr></table>' . "\n";
				echo '</div>' . "\n";

				// --- user / theme / plugin table ---
				echo '<table class="input-table"><tr>' . "\n";

					// --- get test user ---
					$readonly = false;
					$user = $userid = '';
					if ( defined( 'PARAMEDIC_USER' ) ) {
						$user = PARAMEDIC_USER; $readonly = true;
					} elseif ( isset( $_REQUEST['user'] ) ) {
						$user = sanitize_text_field( $_REQUEST['user'] );
					}
					if ( $paramedic['authed'] && ( $user != '' ) ) {
						if ( absint( $user ) == $user ) {
							$userid = $user;
							$user = 'userid';
						} elseif ( ( $user == 'userid' ) && isset( $_REQUEST['userid'] ) ) {
							$userid = sanitize_text_field( $_REQUEST['userid'] );
							$user = 'userid';
						} else {
							$user = '';
						}
					}
					$valid = array( '', 'guest', 'userid' );
					if ( !in_array( $user, $valid ) ) {
						$user = '';
					}

					// --- test user options ---
	                $options = array();
	                if ( is_user_logged_in() ) {
						$options['current'] = __( 'Current User', 'paramedic' );
					}
	                $options['guest'] = __( 'Guest', 'paramedic' ) . " (" . __( 'Logged Out','paramedic') . ")";
					if ( $paramedic['authed'] ) {
						$options['userid'] = __( 'Specific User ID', 'paramedic' );
					}

					// --- test user selector ---
					echo '<td align="center"><b>' . esc_html( __( 'Test User', 'paramedic' ) ) . '</b><br>' . "\n";

						if ( $readonly ) {
							// --- hidden input for readonly ---
							echo '<input type="hidden" name="user" value="' . esc_attr( $user ) . '">';
							if ( ( '' == $user ) || ( 'current' == $user ) ) {
								echo esc_html( __( 'Current User', 'paramedic' ) );
							} elseif ( 'guest' == $user ) {
								echo esc_html( __( 'Guest', 'paramedic' ) ) . " (" . esc_html( __( 'Logged Out', 'paramedic' ) ) . ")";
							} elseif ( 'userid' == $user ) {
								echo esc_html( __( 'User ID', 'paramedic' ) ) . ": " . esc_html( $userid );
							}
						} else {
							// --- user selector ---
							echo '<select name="user" id="user" onchange="paramedic_check_user();">' . "\n";
							foreach ( $options as $key => $label ) {
								echo '<option value="' . esc_attr( $key ) . '"';
								if ( $user == $key ) {
									echo ' selected="selected"';
								}
								echo '>' . esc_html( $label ) . '</option>' . "\n";
							}
							echo '</select><br>' . "\n";

							// --- user ID selection ---
							echo '<div id="userid-wrap"';
							if ( 'userid' != $user ) {
								echo ' style="display:none;"';
							}
							echo '>' . "\n";
								if ( !$paramedic['authed'] ) {
									echo '<div class="help-text">' . esc_html( __( 'Access Key is required', 'paramedic' ) ) . '<br>';
									echo esc_html( __( 'to set Test User.', 'paramedic' ) ) . '</div>' . "\n";
								} else {
									echo '<table><tr>';
										echo '<td>' . esc_html( __( 'User ID', 'paramedic' ) ) . ':</td>';
										echo '<td><input type="text" name="userid" id="userid" value="' . esc_attr( $userid ) . '" style="width:30px;"></td>' . "\n";
										// echo '<td><a href="javascript:void(0);" class="link" onclick="paramedic_find_user();">' . esc_html( __( 'Find', 'paramedic' ) ) . '</a>';
									echo '</tr></table>' . "\n";
								}
							echo '</div>' . "\n";
						}
					echo '</td>' . "\n";

					// --- spacer ---
					echo '<td width="20"></td>' . "\n";

					// --- plugin test options ---
					// 0.9.4: added no plugin test option
					$options = array(
						'none'		=> __( 'No Plugin Tests', 'paramedic' ),
						'active'	=> __( 'Active Plugins', 'paramedic' ),
						'inactive'	=> __( 'Inactive Plugins', 'paramedic' ),
						'network'	=> __( 'Network Active (Multisite)', 'paramedic' ),
						'all'		=> __( 'All Plugins', 'paramedic'),
						// 'custom'	=> __( 'Custom Selection', 'paramedic' ),
					);
					if ( isset( $_REQUEST['plugintest'] ) && array_key_exists( sanitize_text_field( $_REQUEST['plugintest'] ), $options ) ) {
						$current = $_REQUEST['plugintest'];
					// } elseif ( isset( $savedtest['plugintest']) && array_key_exists( $savedtest['plugintest'], $options ) ) {
					//	$current = $savedtest['plugintest'];
					} else {
						$current = 'active';
					}

					// --- plugin test selector ---
					echo '<td align="center"><b>' . esc_html( __( 'Plugins Tests','paramedic' ) ) . '</b><br>' . "\n";

						echo '<select name="plugintest" id="plugintest" onchange="paramedic_check_plugins();">' . "\n";
						foreach ( $options as $option => $label ) {
							echo '<option value="' . esc_attr( $option ) . '"';
							if ( $option == $current ) {
								echo ' selected="selected"';
							}
							echo '>' . esc_html( $label ) . '</option>' . "\n";
						}
						echo '</select><br>' . "\n";

	                    // --- custom plugin selection ---
						echo '<div id="plugin-wrap"';
						if ( 'custom' != $current ) {
							echo ' style="display:none;"';
						}
						echo '>' . "\n";
							echo '<span class="help-text">Coming Soon: Plugin Selector</span>' . "\n";
							// TODO: custom plugin multi-selector
							// echo '<select name="" multiple="multiple">' . "\n";
							// foreach ( $plugins as $basename => $plugin ) {
							//	echo '<option value="' . esc_attr( $basename ) . '"';
							//	if ( in_array( $basename, $selected ) ) {
							//		echo ' selected="selected"';
							//	}
							//	echo '>' . esc_html( $plugin->title ) . '</option>' . "\n";
							// }
							// echo '</select>' . "\n";
						echo '</div>' . "\n";

					echo '</td>' . "\n";

					// --- spacer ---
					echo '<td width="20"></td>' . "\n";

					// --- theme test options ---
					$options = array(
						// TODO: add parent/child theme test options ?
						'none'		=> __( 'No Theme Tests', 'paramedic' ),
						'active'	=> __( 'Active Theme', 'paramedic' ),
						'inactive'	=> __( 'Inactive Themes', 'paramedic' ),
						'network'	=> __( 'Blog Active (Multisite)', 'paramedic' ),
						'all'		=> __( 'All Themes', 'paramedic' ),
						// 'custom'	=> __('Custom Selection','paramedic'),
					);
					if ( isset( $_REQUEST['themetest'] ) && array_key_exists( sanitize_text_field( $_REQUEST['themetest'] ), $options ) ) {
						$current = sanitize_text_field( $_REQUEST['themetest'] );
					// } elseif ( isset( $savedtest['themetest'] ) && array_key_exists( $savedtest['themetest'], $options ) ) {
					//	$current = $savedtest['themetest'];
					} else {
						$current = 'active';
					}

					// --- theme test selector ---
					echo '<td align="center"><b>' . __( 'Theme Tests', 'paramedic' ) . '</b><br>' . "\n";

						echo '<select name="themetest" id="themetest" onchange="paramedic_check_themes();">' . "\n";
						foreach ( $options as $option => $label ) {
							echo '<option value="' . esc_attr( $option ) . '"';
							if ( $option == $current ) {
								echo ' selected="selected"';
							}
							echo '>' . esc_html( $label ) . '</option>' . "\n";
						}
						echo '</select><br>' . "\n";

						// --- custom theme selection ---
						echo '<div id="theme-wrap"';
						if ( 'custom' != $current ) {
							echo ' style="display:none;"';
						}
						echo '><table><tr>' . "\n";
							// TODO: custom theme multi-selector
							echo '<span class="help-text">Coming Soon: Theme Selector</span>' . "\n";
							// echo '<select multiple></select>';
						echo '</tr></table></div>' . "\n";

					echo '</td>' . "\n";

					// --- spacer ---
					echo '<td width="20"></td>' . "\n";

					// --- test all options ---
					$options = array(
						''			=> __( 'Off', 'paramedic' ),
						'active'	=> __( 'Active Resources', 'paramedic' ),
						'inactive'	=> __( 'Inactive Resources', 'paramedic' ),
						'network'	=> __( 'Network Active', 'paramedic' ),
						'all'		=> __( 'All Resources', 'paramedic' ),
					);
					if ( isset( $_REQUEST['testall'] ) && array_key_exists( sanitize_text_field( $_REQUEST['testall'] ), $options ) ) {
						$current = sanitize_text_field( $_REQUEST['testall'] );
					// } elseif ( isset( $savedtest['testall'] ) && array_key_exists( $testall['plugintest'], $options ) ) {
					//	$current = $savedtest['testall'];
					} else {$current = '';}

					// --- test all selector ---
	                echo '<td align="center"><b>' . esc_html( __('All Resources Test', 'paramedic') ) . '</b>';
	                $title = __( 'Overrides plugin and theme test selections.', 'paramedic' );
					echo '<span class="helper" title="' . esc_attr( $title ) . '"><b>?</b></span><br>';
						echo '<select name="testall" id="testall">';
						foreach ( $options as $option => $label ) {
							echo '<option value="' . esc_attr( $option ) . '"';
							if ( $option == $current ) {
								echo ' selected="selected"';
							}
							echo '>' . esc_html( $label ) . '</option>' . "\n";
						}
						echo '</select><br>' . "\n";
					echo '</td>' . "\n";

					// --- plugin test options ---
					// TODO: check values/defines for these options)
					// &repair=				yes or 1
					// &usetheme=			themeslug
					// &useplugins=			pluginslugs
					// &inverse=			yes/1, no/0 or all

					// --- submit button ---
					echo '<td width="20"></td><td align="center">' . "\n";
						echo '<input type="submit" id="test-submit" value="' . esc_attr( __( 'Start Tests Now', 'paramedic' ) ) . '">' . "\n";
					echo '</td>' . "\n";

				echo '</tr></table>' . "\n";

			echo '</form>' . "\n";

			// --- utils buttons ---
			// TODO: view/cleanup test reports, self destruct

		echo '</center></div><br>' . "\n";

		// --- set displayed interface switch ---
		$paramedic['interface'] = true;
	 }
}


// -----------------------------
// === Test Request Handlers ===
// -----------------------------

// -----------------------
// Start Multiple URL Test
// -----------------------
if ( !function_exists( 'paramedic_do_test' ) ) {
	function paramedic_do_test( $testtype, $testfocus ) {

		global $paramedic;
		$paramedic['themes'] = paramedic_get_theme_data();
		$paramedic['plugins'] = paramedic_get_plugin_data();

		// --- manual debug ---
		// print_r($paramedic['themes']);
		// print_r($paramedic['plugins']);
		// print_r($paramedic['networkplugins']);

		// --- set time limit ---
		// (for token expiry and execution time)
		$paramedic['timelimit'] = 600; // 10 minutes default max test time
		if ( defined( 'PARAMEDIC_TIME' ) ) {
			$testtime = absint( PARAMEDIC_TIME );
		} elseif ( isset( $_REQUEST['timelimit'] ) ) {
			$testtime = absint( $_REQUEST['timelimit'] );
		}
		if ( isset( $testtime ) && ( $testtime > 0 ) ) {
			$paramedic['timelimit'] = $testttime;
		}

		// --- create token and set token transient ---
		// 0.9.4: overwrite existing token key for specified test ID if constant is set
		if ( defined( 'PARAMEDIC_TOKEN' ) ) {
			$paramedic['token'] = paramedic_create_token( PARAMEDIC_TOKEN );
		} else {
			$paramedic['token'] = paramedic_create_token();
		}

		// --- set base site test URL ---
		$paramedic['testurl'] = site_url( '/' ); // default to site home
		if ( defined( 'PARAMEDIC_URL' ) ) {
			$testurl = PARAMEDIC_URL;
		} elseif ( isset( $_REQUEST['testurl'] ) ) {
			// 0.9.6: added sanitize_text_field to URL request
			$testurl = trim( sanitize_text_field( $_REQUEST['testurl'] ) );
		}
		if ( isset( $testurl ) && ( '' != $testurl ) ) {
			$paramedic['testurl'] = $testurl;
		}

		// --- maybe set automatic plugin deactivation switch ---
		// 0.9.4: simplified repair value setting
		$paramedic['repair'] = false;
		if ( defined( 'PARAMEDIC_REPAIR' ) ) {
			$paramedic['repair'] = PARAMEDIC_REPAIR;
		} elseif ( isset($_REQUEST['repair'] ) ) {
			if ( in_array( sanitize_text_field( $_REQUEST['repair'] ), array( 'yes', '1' ) ) ) {
				$paramedic['repair'] = true;
			}
		}

		// --- maybe prepare authentication cookies ---
		// 0.9.4: change user settings key check (from testguest)
		if ( !isset( $paramedic['cookies'] ) && ( 'guest' != $paramedic['user'] ) ) {

			$cookies = array();
			if ( 'current' == $paramedic['user'] ) {

				// --- check if user is logged in ---
				paramedic_recheck_user_auth();
				if ( is_user_logged_in() ) {

					// --- get current user cookie values by keys ---
					// 0.9.4: added missing quotes to cookie key contants
					$cookiekeys = array( 'USER_COOKIE', 'PASS_COOKIE', 'AUTH_COOKIE', 'SECURE_AUTH_COOKIE', 'LOGGED_IN_COOKIE' );
					foreach ( $cookiekeys as $cookiekey ) {
						if ( isset( $_COOKIE[$cookiekey] ) ) {
							$cookies[] = array( 'name' => $cookiekey, 'value' => $_COOKIE[$cookiekey] );
						}
		            }
		            // paramedic_debug( "Current User Cookies: " . print_r( $cookies,true ) . PHP_EOL );
				}

			} elseif ( absint( $paramedic['user'] ) > 0 ) {

				// --- get auth cookies for specific user ID ---
				// 0.9.4: auto-generate test auth cookies for user ID
				$cookies = paramedic_get_auth_cookies( $paramedic['user'] );
			}

			if ( count( $cookies ) > 0 ) {
				$paramedic['cookies'] = $cookies;

				// --- set cookie string for Curl ---
				$cookiestring = '';
				foreach ( $cookies as $cookie ) {
					if ( '' != $cookiestring ) {
						$cookiestring .= '&';
					}
					$cookiestring .= $cookie['name'] . '=' . $cookie['value'];
				}
				$paramedic['cookiestring'] = $cookiestring;
			}
		}

		if ( 'core' == $testtype ) {

			// --- just set URL for a single core test ---
			$testurls['core'] = add_query_arg( 'testcore', 'yes', $paramedic['testurl'] );

			// TODO: option to skip loading of network activated plugins ?
			// if (isset($_REQUEST['nonetwork'])) {
			//	if ( ($_REQUEST['nonetwork'] == '1') || ($_REQUEST['nonetwork'] == 'yes') ) {
			//		$testurls['core'] = add_query_arg('nonetworkplugins', 'yes', $testurls['core']);
			//	}
			// }

		} elseif ( 'muplugins' == $testtype ) {

			// TODO: add mu-plugins only test ?

		} elseif ( 'plugins' == $testtype ) {

			// --- set plugin test URLs ---
			// (network / active / inactive / all)
			$testurls = $testplugins = $testnetworkplugins = array();

			// --- network activated plugins on multisite ---
			if ( is_multisite() && ( 'network' == $testfocus ) ) {
				foreach ( $paramedic['networkplugins'] as $plugin ) {
					$testurls[$plugin] = add_query_arg( 'testplugin', $plugin, $paramedic['testurl'] );
					// TODO: maybe use this argument to handle network tests differently ?
					$testurls[$plugin] = add_query_arg( 'networkactive', 'yes', $testurls[$plugin] );
					$testnetworkplugins[] = $plugin;
				}
			}

			// --- active plugins ---
			if ( 'active' == $testfocus ) {
				foreach ( $paramedic['plugins']['active'] as $plugin ) {
					$testurls[$plugin] = add_query_arg( 'testplugin', $plugin, $paramedic['testurl'] );
					$testplugins[] = $plugin;
				}
			}

			// --- inactive plugins ---
			if ( 'inactive' == $testfocus ) {
				foreach ( $paramedic['plugins']['inactive'] as $plugin ) {
					$testurls[$plugin] = add_query_arg( 'testplugin', $plugin, $paramedic['testurl'] );
					$testplugins[] = $plugin;
				}
			}

			// --- all installed plugins ---
			if ( 'all' == $testfocus ) {
				foreach ( $paramedic['plugins']['data'] as $plugin => $data ) {
					$testurls[$plugin] = add_query_arg( 'testplugin', $plugin, $paramedic['testurl'] );
					$testplugins[] = $plugin;
				}
			}

			// TODO: maybe test all of the plugins combined ?
			// if ( count( $testplugins ) > 0 ) {
			//	$pluginstring = implode( ',', $testplugins );
			//	$testurls['COMBINED'] = add_query_arg( 'testplugin', $pluginstring, $paramedic['testurl'] );
			// }

			// --- allow for single or multiple plugin tests (via plugin slugs) ---
			$checkedvalues = array( 'network', 'active', 'inactive', 'all' );
			if ( !in_array( $testfocus, $checkedvalues ) ) {
				if ( count( $paramedic['plugins']['data'] ) > 0 ) {
					if ( strstr( $testfocus, ',' ) ) {
						$pluginstotest = explode( ',', $testfocus );
					} else {
						$pluginstotest = array( $testfocus );
					}

					// --- create a plugin slug index (from directory names) ---
					foreach ( $paramedic['plugins']['data'] as $pluginpath => $plugindata ) {
						if ( strstr( $pluginpath, '/' ) ) {
							$pos = strpos( $pluginpath, '/' );
							$pluginslug = substr( $pluginpath, 0, $pos );
							$pluginslugs[$pluginslug] = $pluginpath;
						} elseif ( '.php' == substr( $pluginpath, -4, 4 ) ) {
							$pluginslug = str_replace( '.php', '', $pluginpath );
							$pluginslugs[$pluginslug] = $pluginpath;
						}
					}

					// --- loop plugin slugs to find matches ---
					foreach ( $pluginstotest as $plugintotest ) {
						if ( array_key_exists( $plugintotest, $pluginslugs ) ) {
							$plugin = $pluginslugs[$plugintotest];
							$testurls[$plugin] = add_query_arg( 'testplugin', $plugin, $paramedic['testurl'] );
						}
					}
				}
			}

		} elseif ( 'themes' == $testtype ) {

			// --- set theme test URLs  ---
			// (active / inactive / all)
			$testurls = array();
			if ( ( 'active' == $testfocus ) || ( 'all' == $testfocus ) ) {
				$activetheme = $paramedic['themes']['active'];
				$testurls[$activetheme] = add_query_arg( 'testtheme', $activetheme, $paramedic['testurl'] );
			}
			if ( ( 'inactive' == $testfocus ) || ( 'all' == $testfocus ) ) {
				foreach ($paramedic['themes']['inactive'] as $theme) {
					$testurls[$theme] = add_query_arg( 'testtheme', $theme, $paramedic['testurl'] );
				}
			}
			// 0.9.4: added network active themes
			if ( 'network' == $testfocus ) {
				foreach ( $paramedic['themes']['network'] as $theme ) {
					$testurls[$theme] = add_query_arg( 'testtheme', $theme, $paramedic['testurl'] );
				}
			}

			// --- allow for single or multiple theme tests (via theme slugs) ---
			$checkedvalues = array( 'active', 'inactive', 'network', 'all' );
			if ( !in_array( $testfocus, $checkedvalues ) ) {
				if ( strstr( $testfocus, ',' ) ) {
					$themestotest = explode( ',', $testfocus );
				} else {
					$themestotest = array( $testfocus );
				}

				if ( array_key_exists( $testfocus, $paramedic['themes']['data'] ) ) {
					foreach ( $themestotest as $theme ) {
						$testurls[$theme] = add_query_arg( 'testtheme', $theme, $paramedic['testurl'] );
					}
				}
			}

		}

		if ( count( $testurls ) > 0 ) {

			// --- add test querystring args to all test URLs ---
			foreach ( $testurls as $testkey => $testurl ) {

				// --- add matching test token and test ID ---
				$testurl = add_query_arg( 'testtoken', $paramedic['token'], $testurl );
				$testurl = add_query_arg( 'testid', $paramedic['testid'], $testurl );

				// --- maybe add usetheme argument (for plugin test override) ---
				if ( isset( $paramedic['usetheme'] ) ) {
					$testurl = add_query_arg( 'usetheme', $paramedic['usetheme'], $testurl );
				}

				// --- maybe passthrough debug switch ---
				if ( $paramedic['debug'] ) {add_query_arg( 'debug', 'yes', $testurl );}

				$testurls[$testkey] = $testurl;
			}

			// --- maybe add all inverse plugins test URLs ---
			// 0.9.1: added inverse plugin testing
			// 0.9.4: only do this if testing inverse of all plugins
			if ( ( 'all' == $paramedic['plugins']['testinverse'] ) && ( 'plugins' == $testtype ) ) {
				$alltesturls = array();
				foreach ( $testurls as $testkey => $testurl ) {
					$alltesturls[$testkey] = $testurl;
					$alltesturls[$testkey] = add_query_arg( 'inversetest', 'yes', $testurl );
				}
				$testurls = $alltesturls;
			}

		    // --- maybe add debug log start marker (once per test instance) ---
		    $thistesttype = strtoupper( substr( $testtype,0,1 ) );
		    $thistesttype .= substr( $testtype,1, strlen( $testtype ) );
		    $paramedic['startmarker'] = "### ParaMedic " . $thistesttype . " Test Start - ID: " . $paramedic['testid'] . ' ###';
		    error_log( $paramedic['startmarker'] . PHP_EOL, 3, $paramedic['logpath'] );

		    // --- do multiple URL test ---
		    $test = paramedic_multi_url_test( $testurls, $testtype, $testfocus );

		    // --- set test end marker ---
		    $paramedic['endmarker'] = "### ParaMedic " . $thistesttype . " Test End - ID: " . $paramedic['testid'] . ' ###';
		    error_log( $paramedic['endmarker'] . PHP_EOL . PHP_EOL, 3, $paramedic['logpath'] );

			// --- maybe output tester debug info ---
			paramedic_debug(
				"Test Type: " . $testtype . PHP_EOL .
				"Test Focus: " . $testfocus . PHP_EOL .
				"Test URLs: " . print_r( $testurls,true ) . PHP_EOL.
				"Test Results: " . print_r( $test,true ) . PHP_EOL
			);

			return $test;
		}

		return false;
	}
}

// -------------------------
// Multiple URL Test Results
// -------------------------
if ( !function_exists( 'paramedic_multi_url_test' ) ) {
	function paramedic_multi_url_test( $testurls, $testtype, $testfocus ) {

		global $paramedic;

		// --- maybe set test method ---
		if ( !isset( $paramedic['method'] ) ) {

			/* $method = 'get';
			if ( isset($_REQUEST['postdata'] ) ) {$method = 'post';}
			if ( extension_loaded( 'curl' ) ) {
				if ( count( $testurls ) == 1 ) {$method = 'single_curl';}
				elseif ( count( $testurls ) > 1 ) {$method = 'multi_curl';}
			} */

			// --- allow for method overrides ---
			// 0.9.1: added overrides via constant or querystring
			// 0.9.3: added validation of possible method values
			// 0.9.4: set method setting key value directly
			if ( defined( 'PARAMEDIC_METHOD' ) ) {$testmethod = PARAMEDIC_METHOD;}
			elseif ( isset( $_REQUEST['method'] ) ) {$testmethod = $_REQUEST['method'];}
			else {$testmethod = 'wp';}

			$valid = array( 'wp', 'curl', 'multicurl' );
			if ( !in_array( $testmethod, $valid ) ) {
				// --- default to WP method ---
				$testmethod = 'wp';
			}
			if ( isset( $testmethod ) ) {
				if ( 'wp' == $testmethod ) {
					if ( isset( $_REQUEST['postdata'] ) ) {$paramedic['method'] = 'post';}
		            else {$paramedic['method'] = 'get';}
		            add_action( 'http_api_curl', 'paramedic_wp_adjust_timeout', 100, 1 );
				} elseif ('curl' == $testmethod ) {
					$paramedic['method'] = 'single_curl';
				} elseif ( 'multicurl' == $testmethod ) {
		            // 0.9.4: temporarily disabled multi_curl method
		            // $paramedic['method'] = 'single_curl';
		            // 0.9.5: re-enabled multi_curl now fatal error logging working
					$paramedic['method'] = 'multi_curl';
				}
			}
		}

		// --- set filtered timeout value ---
		$paramedic['timeout'] = apply_filters( 'paramedic_timeout', 15 );

		// --- start test now ---
		$test = false;
		if ( ( 'get' == $paramedic['method'] ) || ( 'post' == $paramedic['method'] ) ) {

			// --- loop single test URLs using wp_remote_{method} ---
			foreach ( $testurls as $testkey => $testurl ) {

				// --- maybe set authentication cookie objects ---
				if ( isset( $paramedic['cookies'] ) && is_array( $paramedic['cookies'] ) ) {

					// --- maybe load WP cookie class ---
					if ( !class_exists( 'WP_Http_Cookie' ) ) {
						include_once ABSPATH . WPINC . '/class-wp-http-cookie.php';
					}

					// --- convert cookies array to cookie objects ---
					if ( !isset( $paramedic['cookieobjects'] ) ) {
						$cookieobjects = array();
						$cookies = $paramedic['cookies'];
						foreach ( $cookies as $i => $cookie ) {
							$cookieobjects[$i] = new WP_Http_Cookie( $cookie );
						}
						$paramedic['cookieobjects'] = $cookieobjects;
					}

					// --- do single URL request with authentication ---
		            // 0.9.1: handle remote get or post methods
		            // 0.9.5: use filtered timeout value and add args filter
		            $args = array( 'timeout' => $paramedic['timeout'], 'cookies' => $cookieobjects );
		            $args = apply_filters( 'paramedic_remote_args', $args );
					if ( 'get' == $paramedic['method'] ) {
						$result = wp_remote_get( $testurl, $args );
					} elseif ( 'post' == $paramedic['method'] ) {
						// 0.9.1: add post data arguments
						// 0.9.4: added isset check for undefined index warning
						if ( isset( $_REQUEST['postdata'] ) )
							{$args['body'] = $_REQUEST['postdata'];
						}
						$result = wp_remote_post( $testurl, $args );
					}

				} else {

					// --- do single URL request without authentication ---
		            // 0.9.1: handle remote get or post methods
		            // 0.9.5: added timeout and args filter
		            $args = array( 'timeout' => $paramedic['timeout'] );
		            $args = apply_filters( 'paramedic_remote_args', $args );
					if ( 'get' == $paramedic['method'] ) {
						$result = wp_remote_get( $testurl );
					} elseif ( 'post' == $paramedic['method'] ) {
						// 0.9.1: add post data arguments
						// 0.9.4: added isset check for undefined index warning
						if ( isset( $_REQUEST['postdata'] ) ) {
							$args['body'] = $_REQUEST['postdata'];
						}
						$result = wp_remote_post( $testurl, $args );
					}
				}

				// --- get errors ---
				if ( is_wp_error($result ) ) {
					$errors = $result->get_error_messages(); $httpcode = 0;
					// echo "Error when testing ".$testtype.": ".$testkey."<br>".PHP_EOL;
					$error = implode( ', ', $errors );
				} else {
					$error = false;
					$httpcode = wp_remote_retrieve_response_code( $result );
				}

				// echo "Response " . $httpcode . " for " . $testtype . " test: " . $testkey . "<br>" . PHP_EOL;
				$test[] = array(
					'resource'  => $testkey,
					'url'       => $testurl,
					'method'    => $paramedic['method'],
					'httpcode'  => $httpcode,
					'error'     => $error,
				);

			}

		} elseif ( $paramedic['method'] == 'single_curl' ) {

			// --- maybe set authentication cookie string ---
			$cookiestring = false;
			if ( isset( $paramedic['cookiestring'] ) ) {
				$cookiestring = $paramedic['cookiestring'];
			}

			// --- loop single request URLs using curl ---
			foreach ( $testurls as $testkey => $testurl ) {

				// 0.9.1: pass post data argument to CURL
				// 0.9.4: added isset check for undefined index warning
				$postdata = false;
				if ( isset( $_REQUEST['postdata'] ) ) {
					$postdata = $_REQUEST['postdata'];
				}
				$result = paramedic_curl_request( $testurl, $cookiestring, $postdata );
				$test[] = array(
					'resource'  => $testkey,
					'url'       => $testurl,
					'method'    => $paramedic['method'],
					'httpcode'  => $result['code'],
					'error'     => $result['error'],
				);
			}

		} elseif ( $paramedic['method'] == 'multi_curl' ) {

			// TEMP: disabled until error handler can be improved
			// (curl multi works fine, but concurrent logging can confuse results!)

			// --- maybe set authentication cookie string ---
			$cookiestring = false;
			if ( isset( $paramedic['cookiestring'] ) ) {
				$cookiestring = $paramedic['cookiestring'];
			}

			// --- loop multiple request URLs using multi curl ---
			$k = 0; $numurls = count($testurls);
			$threads = apply_filters( 'paramedic_multicurl_threads', 20 );

			// --- convert string indexes to numeric for looping ---
			$i = 0;
			foreach ( $testurls as $testkey => $testurl ) {
				$testurlkeys[$i] = $testkey;
				$testurlvalues[$i] = $testurl;
				$i++;
			}

			while ( $k < $numurls ) {

				// --- set number of threads to do now ---
				$numtodo = $threads;
				if ( ( $numurls - $k ) < $threads ) {
					$numtodo = $numurls - $k;
				}

				// --- initialize multi curl handle ---
				$ch = curl_multi_init();

				// --- loop to add URLs to handle ---
				$result = $curl = $testkeys = array();
				for ( $j = 0; $j < $numtodo; $j++ ) {
					$curl[$j] = paramedic_add_curl_handle( $ch, $testurlvalues[$k], $cookiestring );
					$testkeys[$j] = $testurlkeys[$k];
					$k++;
				}

				// --- execute multi handle ---
				paramedic_exec_multi_handle( $ch );

				// --- loop test results ---
				for ( $j = 0; $j < $numtodo; $j++ ) {

					// --- get curl result ---
					$error = false;
					$content = curl_multi_getcontent( $curl[$j] );
					$httpcode = curl_getinfo( $curl[$j], CURLINFO_HTTP_CODE );
					if ( curl_errno( $curl[$j] ) ) {$error = curl_error( $curl[$j] );}

					// --- set test result data ---
					$test[] = array(
						'resource'  => $testurlkeys[$j],
						'url'       => $testurlvalues[$j],
						'method'    => $paramedic['method'],
						'httpcode'  => $httpcode,
						'error'     => $error,
					);
					// echo "Response ".$httpcode." for ".$testtype." test: ".$testkey."<br>".PHP_EOL;

					// --- clear multi curl handle ---
					curl_multi_remove_handle( $ch, $curl[$j] );
					sleep( 0.25 );
				}

				// --- clear curl multi handle ---
				curl_multi_close( $ch );
				$ch = null;
				unset( $ch );
			}
		}

		return $test;
	}
}

// -------------------
// Single Curl Request
// -------------------
if ( !function_exists( 'paramedic_curl_request' ) ) {
	function paramedic_curl_request( $url, $cookiestring = false, $postdata = false ) {

		global $paramedic;

		// --- set Curl options ---
		// 0.9.5: use filtered timeout value
		$curl = curl_init();
		curl_setopt( $curl, CURLOPT_USERAGENT, 'ParaMedic Error Tester' );
		curl_setopt( $curl, CURLOPT_URL, $url );
		curl_setopt( $curl, CURLOPT_RETURNTRANSFER, 1 );
		curl_setopt( $curl, CURLOPT_CONNECTTIMEOUT, 10 );
		curl_setopt( $curl, CURLOPT_TIMEOUT, $paramedic['timeout'] );

		// --- set SSL and port options ---
		if (is_ssl()) {
			curl_setopt( $curl, CURLOPT_SSL_VERIFYHOST, 0 );
			curl_setopt( $curl, CURLOPT_SSL_VERIFYPEER, false );
			curl_setopt( $curl, CURLOPT_PORT, 443 );
		} else {
			curl_setopt( $curl, CURLOPT_PORT, $_SERVER['SERVER_PORT'] );
		}

		// --- maybe set authentication cookie ---
		if ( $cookiestring ) {
			curl_setopt( $curl, CURLOPT_COOKIE, $cookiestring );
		}

		// --- maybe add post data ---
		// 0.9.1: handle post method and post data
		if ( $postdata ) {
			curl_setopt( $curl, CURLOPT_POST, 1 );
			curl_setopt( $curl, CURLOPT_POSTFIELDS, $postdata );
		}

		// --- execute Curl request ---
		$content = curl_exec( $curl );
		// echo "<!-- URL: ".$url." -->";
		// echo "<!-- Content: ".$content." -->";

		// --- get response info ---
		$httpcode = curl_getinfo( $curl, CURLINFO_HTTP_CODE );
		if ( curl_errno( $curl ) ) {$error = curl_error( $curl );} else {$error = false;}

		// --- clear Curl handle ---
		curl_close( $curl ); unset( $curl );

		// --- return result ---
		$result = array( 'content' => $content, 'code' => $httpcode, 'error' => $error );
		return $result;
	}
}

// ---------------------------------------
// Add Handle to Multiple URL Curl Request
// ---------------------------------------
if ( !function_exists( 'paramedic_add_curl_handle' ) ) {
	function paramedic_add_curl_handle( &$curlhandle, $url, $cookiestring = false, $postdata = false ) {

		global $paramedic;

		// --- set Curl options ---
		// 0.9.5: use filtered timeout value
		$curl = curl_init();
		curl_setopt( $curl, CURLOPT_USERAGENT, 'ParaMedic Error Tester' );
		curl_setopt( $curl, CURLOPT_URL, $url );
		curl_setopt( $curl, CURLOPT_RETURNTRANSFER, 1 );
		curl_setopt( $curl, CURLOPT_CONNECTTIMEOUT, 10 );
		curl_setopt( $curl, CURLOPT_TIMEOUT, $paramedic['timeout'] );

		// --- set SSL and port options ---
		if ( is_ssl() ) {
			curl_setopt( $curl, CURLOPT_SSL_VERIFYHOST, 0 );
			curl_setopt( $curl, CURLOPT_SSL_VERIFYPEER, false );
			curl_setopt( $curl, CURLOPT_PORT, 443 );
		} else {
			curl_setopt( $curl, CURLOPT_PORT, $_SERVER['SERVER_PORT'] );
		}

		// --- maybe set authentication cookie (string) ---
		if ( $cookiestring ) {
			curl_setopt( $curl, CURLOPT_COOKIE, $cookiestring );
		}

		// --- maybe add post data ---
		// 0.9.1: handle post method and post data
		if ( $postdata ) {
			curl_setopt( $curl, CURLOPT_POST, 1 );
			curl_setopt( $curl, CURLOPT_POSTFIELDS, $postdata );
		}

		// --- add handle to multi Curl handle ---
		curl_multi_add_handle( $curlhandle, $curl );
		return $curl;
	}
}

// ------------------------------------
// Execute Multiple Handle Curl Request
// ------------------------------------
// 0.9.5: fix to mismatched function exists wrapper
if ( !function_exists('paramedic_exec_multi_handle' ) ) {
	function paramedic_exec_multi_handle( &$curlhandle ) {
		// 0.9.5: update to use curl_multi_select check
		$active = null;
		do {
		    $status = curl_multi_exec( $curlhandle, $active );
		    if ( $active ) {
		    	curl_multi_select( $curlhandle );
		    }
		} while ( $active && ( $status == CURLM_OK ) );
		// 0.9.5: try to make sure multi curl is finished
		while ( curl_multi_info_read( $curlhandle ) !== false ) {
			sleep( 0.1 );
		}
	}
}


// ---------------------------
// === Error Handler Class ===
// ---------------------------

if ( !class_exists( 'ParamedicErrorHandler' ) ) {
	class ParamedicErrorHandler {

		public static $oldhandler = null;

		public static $throwables = array();

		public static $error_types = array(
			E_ERROR => 'E_ERROR', E_WARNING => 'E_WARNING', E_PARSE => 'E_PARSE', E_NOTICE => 'E_NOTICE',
			E_CORE_ERROR => 'E_CORE_ERROR', E_CORE_WARNING => 'E_CORE_WARNING',
			E_COMPILE_ERROR => 'E_COMPILE_ERROR', E_COMPILE_WARNING => 'E_COMPILE_WARNING',
			E_USER_ERROR => 'E_USER_ERROR', E_USER_WARNING => 'E_USER_WARNING', E_USER_NOTICE => 'E_USER_NOTICE',
			E_STRICT => 'E_STRICT', E_RECOVERABLE_ERROR => 'E_RECOVERABLE_ERROR',
			E_DEPRECATED => 'E_DEPRECATED', E_USER_DEPRECATED => 'E_USER_DEPRECATED'
		);

		public function __construct() {

			// --- set error handling method ---
			$oldhandler = set_error_handler( array( $this, 'error_handler' ) );

			// --- maybe store existing error handler ---
			// ref: https://stackoverflow.com/a/27061620/5240159
			if ( !is_null( $oldhandler ) ) {
				self::$oldhandler = $oldhandler;
			}

			// --- set exception handler ---
		    // [disabled] not sure if correct, as this one is actually *causing* errors!
		    // (apparently if you use exception handling, internal fatal errors return null?)
		    // https://bugs.php.net/bug.php?id=54043
		    // set_exception_handler(array($this, 'exception_handler'));

			// --- set shutdown handling method ---
			register_shutdown_function( array( $this, 'shutdown_handler' ) );
			// add_action('shutdown', array($this, 'shutdown_handler', 999);
		}

		// --- Store Error ---
		public static function set_throwable( $number, $text, $file, $line, $log = true ) {

			global $paramedic;

			// --- no output/logging if error message was suppressed using @ ---
			if ( 0 === error_reporting() ) {return;}

			// --- maybe log to error log ---
			$error = array( 'type' => self::$error_types[$number], 'text' => $text, 'file' => $file, 'line' => $line );
			if ( $log && !in_array( $error, self::$throwables ) ) {

				// --- add test resources to debug log message ---
				$message = '';
				if ( isset( $paramedic['resources'] ) && $paramedic['resources'] ) {
					// 0.9.1: set a different marker wrapper for inverse plugin tests
					if ( isset( $paramedic['plugins']['inversetest'] ) && $paramedic['plugins']['inversetest'] ) {
						$plugins = implode( ',', $paramedic['plugins']['test'] );
						$message .= '[-'.$plugins.'-]';
					} else {
						$message .= '[+'.$paramedic['resources'].'+]';
					}
				} else {
					$message .= '[?unknown?]';
				}

				// --- strip ABSPATH from log line for ease of reading ---
				// 0.9.4: strip ABSPATH from text message also
				$file = str_replace( ( substr( ABSPATH, 0, -1 ) ), '', $file );
				$text = str_replace( ( substr( ABSPATH, 0, -1 ) ), '', $text );

				// --- set error message and log it ---
				$message .= self::$error_types[$number] . ': line ' . $line . ' in ' . $file . ': ' . $text . PHP_EOL;
				error_log( $message, 3, $paramedic['logpath'] );

				// --- store error to prevent duplicates ---
				self::$throwables[] = $error;
			}
		}

		// --- Error Handler ---
		public static function error_handler( $number = '', $text = '', $file = '', $line = '', $context = '' ) {

			// --- store error ---
			self::set_throwable( $number, $text, $file, $line, true );

			// --- maybe call stored error handler ---
			// 0.9.4: added for multiple error handler compatibility
			if ( !is_null( self::$oldhandler ) ) {
				call_user_func( self::$oldhandler, $number, $text, $file, $line, $context );
			}

			// --- disable default error reporting ---
			return true;
		}

		// --- Exception Handler ---
		public static function exception_handler( Exception $e ) {
			self::set_throwable( $e->getCode(), $e->getMessage(), $e->getFile(), $e->getLine(), true );
			return true;
		}

		// --- Shutdown Handler ---
		public static function shutdown_handler() {

			// --- get last error ---
			global $paramedic;
		    $e = error_get_last();

		    // --- check and log fatal errors ---
		    $fatal = array( E_ERROR, E_PARSE, E_CORE_ERROR, E_USER_ERROR, E_COMPILE_ERROR, E_RECOVERABLE_ERROR );
		    if ( !is_null( $e ) ) {
		        if ( isset( $e['type'] ) && in_array($e['type'], $fatal ) ) {

		            // --- log fatal error using error handler ---
		            // error_log("Fatal Error: ".print_r($e,true), 3, $paramedic['logpath']);
		            self::set_throwable( $e['type'], $e['message'], $e['file'], $e['line'], true );

		            // --- record resource usage anyway ---
		            paramedic_test_finish();
		        }
		    }
			return false;
		}

	}
}


// ----------------------------
// === Test Result Collater ===
// ----------------------------
// 0.9.1: added testtype argument (for inverse plugin testing)
if (!function_exists('paramedic_collate_errors')) {
	function paramedic_collate_errors( $test, $testtype, $errors = false ) {

		global $paramedic;

		// --- set resource keys from test data ---
		$testresources = $testresults = array();
		foreach ( $test as $testdata ) {
			$testresources[] = $testdata['resource'];
		}

		// --- set existing errors (to filter out) ---
		if ('plugins' == $testtype ) {
			$errorplugins = array();
		}
		if ( $errors && is_array($errors ) ) {
			$filter = array();
			foreach ( $errors as $resource => $error ) {
				if ( is_array( $error ) ) {
					foreach ( $error as $line ) {$filter[] = $line;}
				}
			}
		} else {
			$filter = false;
		}

		// --- set debug log path and test start marker ---
		$logpath = $paramedic['logpath'];
		$startmarker = $paramedic['startmarker'];
		$endmarker = $paramedic['endmarker'];

		// --- maybe debug info ---
		paramedic_debug(
		    "Test Instance Log Path: " . $paramedic['logpath'] . PHP_EOL .
			"Test Resource Keys: " . print_r( $testresources,true ) . PHP_EOL .
		    "Test Start Marker String: ]" . $startmarker . '[' .PHP_EOL .
		    "Test End Marker String: ]" . $endmarker . '[' . PHP_EOL
		);

		// --- check for existing debug log ---
		if ( file_exists( $logpath ) ) {

			// --- read the debug log file... backwards..! ---
			// (so as not to risk running out of memory)
			// Ref: http://stackoverflow.com/a/26595154/5240159

		    $startfound = $endfound = false;
		    $loglines = $bufferlines = array();
			$lines = 0; $maxlines = 9999; // read plenty but not unlimited lines

			// Helper Functions
			// ----------------

			// --- moves cursor one step back ---
			// (if can - returns true, if can't - returns false)
			if (!function_exists('paramedic_movestepback')) {
				function paramedic_movestepback( &$f ) {
					if ( ftell($f) > 0 ) {
						fseek( $f, -1, SEEK_CUR );
						return true;
					} else {
						return false;
					}
				}
			}

			// --- reads $length chars ---
			// (but moves cursor back where it was before reading)
			if (!function_exists('paramedic_readnotseek')) {
				function paramedic_readnotseek( &$f, $length ) {
					$r = fread( $f, $length );
					fseek( $f, -$length, SEEK_CUR );
					return $r;
				}
			}

			// --- open the file ---
			if ( $v = @fopen( $logpath, 'r' ) ) {

				// --- move cursor to the end of the file ---
				fseek( $v, 0, SEEK_END );

				// --- while there is at least 1 character to read ---
				while ( ftell( $v ) > 0 ) {

					$newline = false; $charcounter = 0;

					// --- line counting ---
					// (not start of a line / the file)
					while ( !$newline && paramedic_movestepback( $v ) ) {
		                if ( paramedic_readnotseek( $v, 1 ) == "\n" ) {$newline = true;}
		                // if ( paramedic_readnotseek( $v, 1 ) == PHP_EOL ) {$newline = true;}
						$charcounter++;
					}

					// --- line reading / printing ---
					// (if there was anything on the line)
					if ( $charcounter > 1 ) {

						// --- (unused) prints missing "\n" before last *printed* line ---
						// if ( !$newline ) {echo "<br>";}

						// --- gets current line ---
						$thisline = trim( paramedic_readnotseek( $v, $charcounter ) );

		                // --- maybe output raw log line ---
		                $rawlog = false; // $rawlog = true; // dev debug mode
		                if ( $rawlog ) {
		                	paramedic_debug( "Raw Log Line: " . $thisline . PHP_EOL );
		                }

						// --- finish reading when we reach the test ID marker! ---
						if ( strstr( $thisline, $startmarker ) ) {
							paramedic_debug( "*Test Start Marker Found*" . PHP_EOL );
							// 0.9.4: set marker found switch and break
							$startfound = true; break;
						}

		                // --- start reading when we reach end test marker ---
		                // 0.9.5: added test end marker detection
		                if ( strstr( $thisline, $endmarker ) ) {
		                    paramedic_debug( "*Test End Marker Found*" . PHP_EOL );
		                    $endfound = true;
		                }

		                if ( $endfound ) {

		                    // --- detect new log entry ---
		                    if ( substr( $thisline, 0, 1 ) == '[') {

		                        // --- capture unhandled fatal errors ---
		                        // 0.9.5: removed now that fatal errors are correctly captured
		                        // $trimerrors = array('] PHP Fatal error:', '] PHP Parse error:');
		                        // foreach ($trimerrors as $trimerror) {
		                        //    if (strstr($thisline, $trimerror)) {
		                        //        $pos = strpos($thisline, ']') + strlen($trimerror);
		                        //        $thisline = substr($thisline, $pos, strlen($thisline));
		                        //        $thisline = 'E_ERROR:'.$thisline;
		                        //    }
		                        // }

		                        // --- strip ABSPATH if not already stripped ---
		                        $thisline = str_replace( ( substr(ABSPATH, 0, -1 ) ), '', $thisline );

		                        // --- add log line if not duplicate ---
		                        if ( !in_array( $thisline, $loglines ) ) {
		                            $loglines[] = $thisline;

		                            // --- store buffer lines via log line index ---
		                            // 0.9.4: track extra buffered lines
		                            $lineindex = count( $loglines ) - 1;
		                            if ( count( $bufferlines ) > 0) {
		                                $last = count( $bufferlines ) - 1;
		                                if ( substr( $bufferlines[$last], 0, strlen('Stack trace' ) ) == 'Stack trace') {
		                                    $stacktraces[$lineindex] = $bufferlines;
		                                } else {$extralines[$lineindex] = $bufferlines;}
		                                $bufferlines = array();
		                            }
		                        }
		                    } else {
		                        // --- buffer log lines ---
		                        // 0.9.4: buffer unknown log lines
		                        $bufferlines[] = $thisline;
		                    }
		                }

						$lines++;
					}

					// --- handle limited or unlimited lines ---
					if ( ( $maxlines > 0 ) && ( $lines > $maxlines ) ) {break;}
				}

				// --- close the debug file ---
				fclose( $v );
			}

			// --- maybe process any log lines for this test instance ---
			if ( $startfound && ( count( $loglines) > 0 ) ) {

		        paramedic_debug( "Log Lines: " . print_r( $loglines, true ) );

				// --- reverse log lines preserving keys ---
				$loglines = array_reverse( $loglines, true );

				// --- loop log lines ---
				foreach ( $loglines as $i => $logline ) {

					// --- debug log line output ---
					paramedic_debug( $logline . PHP_EOL );

					// 0.9.1: handle inverse plugin test markers also
					$startwrap = $endwrap = $inverse = $stats = false;
					if ( strstr( $logline, '[+' ) && strstr( $logline, '+]' ) ) {
						$startwrap = '[+'; $endwrap = '+]';
					} elseif ( strstr( $logline, '[-') && strstr( $logline, '-]' ) ) {
						$startwrap = '[-'; $endwrap = '-]'; $inverse = true;
					} elseif ( strstr( $logline, '[*') && strstr( $logline, '*]' ) ) {
						$startwrap = '[*'; $endwrap = '*]'; $stats = true;
					}

					if ( $startwrap && $endwrap ) {

						// --- strip start and end markers to get resource(s) ---
						$posa = strpos( $logline, $startwrap ) + strlen( $startwrap );
						$posb = strpos( $logline, $endwrap );
						$resource = substr( $logline, $posa, ( $posb - $posa ) );
						if ( strstr( $resource, ',' ) ) {
							$resources = explode( ',', $resource );
						} else {
							$resources = array( $resource );
						}
					}

					if ( isset( $resources ) ) {

						// --- match resources with test resources ---
						$intersect = array_intersect( $resources, $testresources );

						// --- maybe debug resource intersection ---
						// paramedic_debug(
						//	"Resources: " . print_r( $resources,true ) . PHP_EOL .
						//	"Test Resources: " . print_r( $testresources,true ) . PHP_EOL .
						//	"Intersection: " . print_r( $intersect,true ) . PHP_EOL
						// );

						// --- loop resource intersection ---
						if ( count( $intersect) > 0 ) {
							foreach ( $intersect as $resourcekey ) {

								// --- remove resource(s) to get errors ---
								$thisline = str_replace( $startwrap . $resource . $endwrap, '', $logline );
								$thisline = str_replace( $resource . ',', '', $thisline );
		                        $thisline = str_replace( ',' . $resource, '', $thisline );
		                        $thisline = str_replace( $startwrap, '', $thisline );
		                        $thisline = str_replace( $endwrap, '', $thisline );
								$thisline = trim($thisline);
								if ( substr( $thisline, 0, 2 ) == 'E_') {
									$thisline = substr( $thisline, 2, strlen( $thisline ) );
								}
								// echo $thisline."<br>";

								if ( !$filter || ( $filter && !in_array( $thisline, $filter ) ) ) {

									// 0.9.1: mark inverse test results via resource key
		                            // 0.9.4: just set key separate to adding log lines
		                            // 0.9.5: handle stats lines as test results
		                            if ( $inverse ) {
		                            	$resultindex = '-' . $resourcekey;
		                            } elseif ( $stats ) {
		                            	$resultindex = '*' . $resourcekey;
		                            } else {
		                            	$resultindex = $resourcekey;
		                            }

									// --- add log line to test results ---
									$testresults[$resultindex][] = $thisline;

									// TODO: maybe capture stack traces to test results
									// if (isset($stacktraces[$i])) {}

									// --- maybe add extra lines to test results ---
									// if (isset($extralines[$i])) {}

									// 0.9.4: store error plugins for inverse plugin test
									if ( !$inverse ) {
										if ( ( 'plugins' == $testtype ) && ( 'yes' == $paramedic['plugins']['testinverse'] ) ) {
											$errorresources = array();
											foreach ( $intersect as $key ) {
												if ( $key != $resourcekey ) {
													$errorresources[] = $key;
												}
											}
											$errorplugins[$resourcekey] = $errorresources;
										}
									}
								}
							}
						}
					}
				}

				// --- debug test results ---
				paramedic_debug( "Test Results: " . print_r( $testresults, true ) . PHP_EOL);

				// --- maybe do single inverse plugin tests ---
				if ( ( 'plugins' == $testtype ) && ( 'yes' == $paramedic['plugins']['testinverse'] ) && ( count( $errorplugins ) > 0 ) ) {
					foreach ( $errorplugins as $errorplugin => $resources ) {

						// TODO: do single inverse plugin tests here..!
						// $results = paramedic_do_test('plugins', $resources);
						// $inversetesturl = add_query_arg('testplugins', $resources', $paramedic['testurl']);
						// $inversetesturl = add_query_arg('inversetest', 'yes', $inversetesturl);

					}
				}

			}

			// --- check if test marker was found ---
			if ( $startfound ) {

				// --- set no-error string if there are no log lines for a resource --
				foreach ( $testresources as $resourcekey ) {
					if ( !isset( $testresults[$resourcekey] ) ) {
						$testresults[$resourcekey] = 'no-errors';
					}

					// 0.9.1: set no-error string for inverse plugin test resources also
					// 0.9.4: only do this when testing inverse of all plugins
					if ( ( 'plugins' == $testtype ) && ( 'all' == $paramedic['plugins']['testinverse'] ) ) {
						if (!isset($testresults['-'.$resourcekey])) {
							$testresults['-'.$resourcekey] = 'no-errors';
						}
					}
				}

				// --- sort test results by key ---
				ksort( $testresults );

			} else {
				// 0.9.4: added test marker not found message
				paramedic_debug( 'Error! Test Marker was not Found!' . PHP_EOL );
			}

		} else {
			// 0.9.4: added log not found message
			paramedic_debug( 'Error! Log not Found: ' . $logpath . PHP_EOL );
		}

		// print_r($testresults);

		return $testresults;
	}
}

// ---------------------------
// === Output Test Results ===
// ---------------------------
if ( !function_exists( 'paramedic_output_results' ) ) {
	function paramedic_output_results( $testresults, $test, $testtype ) {

		global $paramedic;

		// --- recheck user
		paramedic_recheck_user_auth();

		// --- maybe output display header ---
		if ( !$paramedic['silent'] ) {

			// --- change result output font to console style ---
			echo '<div style="font-size:18px; font-family: Consolas, \'Lucida Console\', Monaco, FreeMono, monospace;">';
		}

		// --- loop test results ---
		// 0.9.4: set empty log results string
		$results = '';
		$testresources = array();
		if ( $testresults && is_array( $testresults ) ) {

			// --- set error display colours ---
			// 0.9.4: updated colours for dark scheme
			$errorcolours = array(
				'ERROR' => '#D00', 'PARSE' => '#D00', 'WARNING' => '#FA0',
				'NOTICE' => '#04F', 'STRICT' => '#04F', 'UNKNOWN' => '#A0F'
			);

			// --- maybe start display table ---
			if ( !$paramedic['silent'] ) {
				echo '<table width="100%">';
			}

		    // --- get stats lines ---
		    foreach ( $testresults as $resource => $loglines ) {
		        if ( '*' == substr( $resource, 0, 1 ) ) {
		            unset( $testresults[$resource] );
		            $resource = substr( $resource, 1, strlen( $resource ) );
		            $statsresults[$resource] = $loglines[0];
		        }
		    }

			// 0.9.1: maybe reorder for inverse plugin test results
			if ( 'plugin' == $testtype ) {
				foreach ( $testresults as $resource => $loglines ) {
					if ( '-' == substr( $resource, 0, 1 ) ) {
						unset( $testresults[$resource] );
						$resource = substr( $resource, 1, strlen( $resource ) );
						$inversetestresults[$resource] = $loglines;
					}
				}
				foreach ( $testresults as $resource => $loglines ) {
					$newtestresults[$resource] = $loglines;
					if ( isset( $inversertestresults[$resource] ) ) {
						$newtestresults['-'.$resource] = $inversetestresults[$resource];
					}
				}
				$testresults = $newtestresults;

				// --- debug reordered test results ---
				// paramedic_debug("Reordered Test Results: " . print_r( $newtestresults, true ) . PHP_EOL);

			}

			foreach ( $testresults as $resource => $loglines ) {

				// --- check / set inverse test flag ---
				// 0.9.1: added for inverse tests
				$inverse = false;
				if ( ( 'plugin' == $testtype ) && ( '-' == substr( $resource, 0, 1 ) ) ) {
					$resource = substr( $resource, 1, strlen( $resource ) );
					$inverse = true;
				}

				// --- clear display output ---
				$testdisplay = $resourcedisplay = $versiondisplay = $resultdisplay = '';
				$errordisplay = $codedisplay = $resultlines = $deactivateddisplay = $version = '';

				// --- clear error counts ---
				$errorcount = array(
					'ERROR' => 0, 'PARSE' => 0, 'WARNING' => 0,
					'NOTICE' => 0, 'STRICT' => 0, 'OTHER' => 0
				);

				// --- for core test ---
				if ( 'core' == $testtype ) {
					$code = $test[0]['httpcode'];
					$error = $test[0]['error'];
					$method = $test[0]['method'];
					if ( 200 == $code ) {
						$codecolor = '#0B0';
					} elseif ( ( 500 == $code ) || ( 0 === $code) ) {
						$codecolor = '#D00';
					} else {
						$codecolor = '#EEE';
					}
				} elseif ( ( 'theme' == $testtype ) || ( 'plugin' == $testtype ) ) {
					foreach ( $test as $i => $data ) {
						if ( $data['resource'] == $resource ) {
							$code = $data['httpcode'];
							$error = $data['error'];
							$method = $data['method'];
						}
					}
					if ( 200 == $code ) {
						$codecolor = '#0B0';
					} elseif ( ( 500 == $code ) || ( 0 === $code) ) {
						$codecolor = '#D00';
					} else {
						$codecolor = '#EEE';
					}
				}

				// --- loop debug log lines ---
				if ( is_array( $loglines ) ) {
					foreach ( $loglines as $logline ) {
						$founderrortype = false;
						foreach ( $errorcolours as $errortype => $colour ) {
							if ( substr( $logline, 0, strlen( $errortype ) ) == $errortype ) {
								$logline = substr( $logline, strlen( $errortype ), strlen($logline ) );
								$logline = '<span color="' . $colour . '">' . $errortype . '</span>' . $logline;
								$errorcount[$errortype]++;
								$founderrortype = true;
							}
						}
						if ( !$founderrortype ) {
							$errorcount['OTHER']++;
						}
						$resultlines .= '<div style="font-size:16px;">' . $logline . '</div><br>' . PHP_EOL;
					}

					// --- count fatals warnings and notices ---
					// TODO: maybe add more accurate matching for "other" error codes?
					$fatals = $errorcount['ERROR'] + $errorcount['PARSE'];
					$warnings = $errorcount['WARNING'];
					$notices = $warnings + $errorcount['NOTICE'] + $errorcount['STRICT'] + $errorcount['OTHER'];

					// --- set error count display column ---
					// 0.9.4: updated colours for dark colour scheme
					$errordisplay = '<td>';
						if ( $fatals > 0 ) {
							$colour = "red";
						} else {
							$colour .= "#EEE";
						}
						$errordisplay .= '<b><span color="' . $colour . '">';
						$errordisplay .= $fatals . '</span></b> ' . esc_html( __( 'Errors','paramedic' ) );
						if ( $warnings > 0 ) {
							$colour = "#FA0";
						} elseif ( $notices > 0 ) {
							$colour = "#04F";
						} else {
							$colour = "#EEE";
						}
						$errordisplay .= ' <b><span color="' . $colour . '">';
						$errordisplay .= $notices . '</span></b> ' . esc_html( __('Notices','paramedic') );
					$errordisplay .= '</td>';

					// --- set error result display ---
					$resultdisplay = '<td>';
						if ( $error ) {
							$colour = $errorcolours['UNKNOWN'];
						} elseif ( $fatals > 0 ) {
							$colour = $errorcolours['ERROR'];
						} elseif ( $warnings > 0 ) {
							$colour = $errorcolours['WARNING'];
						} elseif ( $notices > 0 ) {
							$colour = $errorcolours['NOTICE'];
						}
						$resultdisplay .= '<b><div color="' . $colour . '">';
						$resultdisplay .= $code . ' ';
						if ( $error ) {
							$resultdisplay .= esc_html( __( 'UNKNOWN', 'paramedic' ) );
						} elseif ( $fatals > 0 ) {
							$resultdisplay .= esc_html( __( 'FAILED', 'paramedic' ) );
						} else {
							$resultdisplay .= esc_html( __( 'PASSED', 'paramedic' ) );
						}
					$resultdisplay .= "</div></b></td>";

				} else {

					// --- set test passed display ---
					$resultdisplay = '<td><div color="#0B0">' . esc_html( $code ) . ' ' . esc_html( __( 'PASSED','paramedic' ) ) . '</div></td>';
					$errordisplay = '<td>0 Errors 0 Notices</td>';
				}

				if ( $errorcount['ERROR'] > 0 ) {
					$color = $errorcolours['ERROR'];
				} elseif ( $errorcount['PARSE'] > 0 ) {
					$color = $errorcolours['PARSE'];
				} elseif ( $errorcount['WARNING'] > 0 ) {
					$color = $errorcolours['WARNING'];
				} elseif ( $errorcount['NOTICE'] > 0 ) {
					$color = $errorcolours['NOTICE'];
				} else {
					$color = '#0B0';
				}

				if ( 'core' == $testtype ) {

					// --- maybe display core version ---
					if ( $paramedic['authed'] ) {
						global $wp_version, $cp_version;
						if ( isset( $wp_version ) ) {
							$version = 'v' . $wp_version;
						} elseif ( isset( $cp_version ) ) {
							$version = 'v' . $cp_version;
						}
					}

		            // 0.9.5: moved code block from above
		            $testdisplay = '<td><b>' . esc_html( __( 'Error Test', 'paramedic' ) ) . '</b></td>';
					$resourcedisplay = '<td><b><div color="' . $color . '">' . esc_html( __( 'WordPress Core', 'paramedic' ) ) . '</div></b></td>';
					// $codedisplay = '<td>HTTP ' . __( 'Code', 'paramedic' ) . ' <b><div color="' . $codecolor . '">' . $code . '</div></b>';

				} elseif ( 'theme' == $testtype ) {

					// --- maybe display theme version ---
					$version = '';
					// if ($paramedic['authed'] || current_user_can('manange_themes')) {
						foreach ( $paramedic['themes']['data'] as $theme => $data ) {
							if ($theme == $resource) {
								$version = 'v' . $data['Version'];
							}
						}
					// }

					$testdisplay = '<td><b>' . __( 'Theme Test','paramedic' ) . '</b></td>';
					$resourcedisplay = '<td><b><div color="' . $color . '">' . $resource . '</font></b></td>';
					// $codedisplay = '<td>HTTP ' . __( 'Code', 'paramedic' ) . ' <b><div color="' . $codecolor . '">' . $code . '</div></b>';

				} elseif ( 'plugin' == $testtype ) {

					// --- maybe display plugin version ---
					$version = '';
					// if ($paramedic['authed'] || current_user_can('manage_plugins')) {
						foreach ( $paramedic['plugins']['data'] as $plugin => $data ) {
							if ( $plugin == $resource ) {
								$version = 'v' . $data['Version'];
							}
						}
					// }

					if ($inverse) {
						// 0.9.1: set different text output for inverse plugin test
						$testdisplay = '<td><b>' . esc_html( __( 'All Except', 'paramedic' ) ) . '</b></td>';
					} else {
						$testdisplay = '<td><b>' . esc_html( __( 'Plugin Test', 'paramedic' ) ) . '</b></td>';
					}

					// 0.9.1: maybe split for shortened display
					$displayresource = $resource;
					if ( strstr( $resource, '/' ) ) {
						$parts = explode( '/', $resource );
						$displayresource = $parts[0];
					}
					$resourcedisplay = '<td title="' . esc_attr( $resource ) . '"><b><div color="' . $color . '">';
					$resourcedisplay .= $displayresource;
					$resourcedisplay .= '</div></b></td>';

					// $codedisplay = "<td>HTTP ".__('Code','paramedic')." <b><font color='".$codecolor."'>".$code."</font></b>";

					// TEMP: disabled for further testing
					// --- maybe deactivate an active plugin if there were fatal errors ---
					// if ($paramedic['repair'] && ($errorcount['ERROR'] > 0) ) {
					//
					//	// 0.9.1: do not deactivate single plugin in an inverse test
					//	if (!$inverse) {
					//
					//		// --- maybe deactivate a network plugin ---
					//		if (is_multisite() && (is_array($paramedic['networkplugins']))
					//		&& (in_array($resource, $paramedic['networkplugins'])) ) {
					//			foreach ($paramedic['networkplugins'] as $index => $plugin) {
					//				if ($plugin == $resource) {unset($paramedic[$index]);}
					//			}
					//			$paramedic['deactivatedplugins'][] = $resource;
					//			update_site_option('active_network_plugins', $paramedic['networkplugins']);
					//			$deactivateddisplay = __('Network Plugin with Errors DEACTIVATED SITEWIDE','paramedic').": ".$resource;
					//			// --- send message to log file without triggering an error ---
					//			error_log("!!! ".$message, 3, $paramedic['logpath']);
					//		}
					//
					//		// --- maybe deactivate a single plugin ---
					//		if ( (is_array($paramedic['plugins']['active']))
					//		  && (in_array($resource, $paramedic['plugins']['active'])) ) {
					//			foreach ($paramedic['plugins']['active'] as $index => $plugin) {
					//				if ($plugin == $resource) {unset($paramedic['plugins']['active'][$index]);}
					//			}
					//			$paramedic['deactivatedplugins'][] = $resource;
					//			update_option('active_plugins', $paramedic['plugins']['active']);
					//			$deactivateddisplay = __('Plugin with Errors DEACTIVATED','paramedic').": ".$resource;
					//			// --- send message to log file without triggering an error ---
					//			error_log($deactivateddisplay, 3, $paramedic['logpath']);
					//		}
					//	} else {
					//		// TODO: work out what to do for 'repair' with an inverse test?
					//	}
					// }
				}

		        // --- test method display ---
		        // 0.9.5: removed method display
				// $codedisplay .= " ".__('via','paramedic')." ";
				// 0.9.1: added wp_remote_post method
				// if ($method == 'get') {$codedisplay .= "<i>wp_remote_get</i>";}
				// elseif ($method == 'post') {$codedisplay .= "<i>wp_remote_post</i>";}
				// elseif ($method == 'single_curl') {$codedisplay .= "<i>Curl</i>";}
				// elseif ($method == 'multi_curl') {$codedisplay .= "<i>Multi-Curl</i>";}
				// $codedisplay .= '</td>';

				// --- if the test failed, output this in the result lines ---
				if  ( $error ) {
					$resultlines = '! ' . esc_html( __( 'Test Error', 'paramedic' ) ) . ': ' . $error . '<br>' . PHP_EOL . $resultlines;
				}

				// --- version display ---
				$versiondisplay .= '<td>' . esc_html( $version ) . '</td>';

		        // --- stats display ---
		        $statsdisplay = '';
		        if ( isset( $statsresults[$resource] ) ) {
		            $stats = explode( ',', $statsresults[$resource] );
		            $memory = str_replace( 'M:', '', $stats[0] );
		            $memory = __( 'Memory','paramedic' ) . ': ' . round( $memory/1024/1024, 2 ) . 'mb';
		            $queries = str_replace( 'Q:', '', $stats[2] );
		            $queries = __( 'Queries','paramedic' ) . ': ' . $queries;
		            $time = str_replace('T:', '', $stats[1] );
		            if ( 'core' == $testtype ) {
		            	$paramedic['coreloadtime'] = $time;
		            } else {
		                $diff = $time - $paramedic['coreloadtime'];
		                if ( $diff < 0 ) {
		                	$diff = '';
		                } else {
		                	$diff = ' (+' . ( round ( $diff, 3 ) * 1000 ) . ')';
		                }
		            }
		            $time = ( round( $time, 3 ) * 1000 ) . 'ms';
		            $time = __( 'Loadtime', 'paramedic' ) . ': ' . $time;
		            $statsdisplay = $memory . ' ' . $queries . ' ' . $time;
		            if ( 'core' != $testtype ) {
		            	$statsdisplay .= $diff;
		            }
		        }
		        $statsdisplay = '<td><div style="font-size:16px;">' . esc_html( $statsdisplay ) . '</div></td>';

				// --- output table columns for this resource ---
				$output = $testdisplay . $resourcedisplay . $versiondisplay . $resultdisplay . $errordisplay . $statsdisplay;
				if ( !$paramedic['silent'] ) {
					echo '<tr class="result-row">' . $output . '</tr>' . PHP_EOL;
					if ( '' != $deactivateddisplay ) {
						echo '<tr><td colspan="7"><span color="#D00">!!!</span> ';
						echo '<b>' . $deactivateddisplay . '</b>';
						echo ' <span color="#D00"> !!!</span></td></tr>' . PHP_EOL;
					}
					echo '<tr><td colspan="7">' . $resultlines . '</td></tr>' . PHP_EOL;
					echo '<tr height="10"><td> </td></tr>' . PHP_EOL;
				}

				// --- strip HTML for result logging file ---
				$results .= strip_tags( $output ) . PHP_EOL . strip_tags( $resultlines ) . PHP_EOL;
			}

			if ( !$paramedic['silent'] ) {
				echo '</table>';
			}
		}

		if ( !$paramedic['silent'] ) {
			echo '</div>';
			if ( $paramedic['debug'] ) {
				echo 'Resources: ' . print_r( $testresources,true) . '<br>';
			}
		}

		return $results;
	}
}


// ------------------------
// === Helper Functions ===
// ------------------------

// --------------------
// Match Trace Function
// --------------------
if ( !function_exists('paramedic_match_trace' ) ) {
	function paramedic_match_trace( $function, $file, $depth ) {

		// --- manual function debug switch only ---
		$debug = false;

		// --- get function trace ---
		$e = new Exception;
		$trace = $e->getTrace();
		if ( $debug ) {
			echo 'Trace: ' . print_r( $trace,true ) . '<br>';
		}

		// --- loop trace levels ---
		if ( count( $trace ) > 0) {
			foreach ( $trace as $i => $level ) {

				// --- maybe output debug values ---
				if ( $debug ) {
					if ( !isset( $level['file'] ) ) {
						$level['file'] = '';
					}
					$level['file'] = str_replace( ABSPATH, '', $level['file'] );
					echo "Function Match: " . $function . " - Function " . ( $i + 1 ) . ": " . $level['function'];
					echo " - File Match: " . $file . " - File: " . $level['file'] . "<br>";
				}

				// --- check if function and file match ---
				if ( isset( $level['function'] ) && ( $function == $level['function'] ) && isset( $level['file'] )
				  && ( substr( $level['file'], -strlen( $file ), strlen( $file ) ) == $file ) ) {
					return true;
				}

				// --- check depth ---
				if ( ( $i + 1 ) > $depth ) {
					return false;
				}
			}
		}
		return false;
	}
}

// -----------------------
// Create Error Test Token
// -----------------------
if ( !function_exists( 'paramedic_create_token' ) ) {
	function paramedic_create_token( $token = false ) {

		global $paramedic;

		// --- set time limit to match token expiry ---
		@set_time_limit( $paramedic['timelimit'] );

		// --- generate test token ---
		if ( !$token ) {
			if ( !function_exists( 'wp_generate_password' ) ) {
				include_once ABSPATH . WPINC . '/pluggable.php';
			}
			$token = wp_generate_password( 12, false, false );
		}

		// --- set the test token to transient option ---
		set_transient( 'paramedic_token_' . $paramedic['testid'], $token, $paramedic['timelimit'] );
		return $token;
	}
}

// ---------------------------------
// Log Results to Test Instance File
// ---------------------------------
if (!function_exists('paramedic_log_results')) {
	function paramedic_log_results( $results ) {

		global $paramedic;

		// --- bug out if test logging is disabled ---
		if ( !$paramedic['testlog'] ) {return;}

		// --- set test result log path ---
		// 0.9.3: fix to use filepath based on directory (as debug filename may not be debug.log)
		$logpath = dirname( $paramedic['logpath'] ) . '/paramedic-test-' . $paramedic['testid'] . '.log';

		// --- use error_log to log result line ---
		error_log( $results, 3, $paramedic['logpath'] );
	}
}

// --------------------------
// Set All Plugin Global Data
// --------------------------
if ( !function_exists('paramedic_get_plugin_data' ) ) {
	function paramedic_get_plugin_data() {

		global $paramedic;

		$paramedic['plugins']['test'] = array();

		// --- get plugins and sort alphabetically ---
		if ( !function_exists('get_plugins' ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$pluginsdata = paramedic_get_plugins();
		// print_r($pluginsdata);
		$pluginkeys = array_keys( $pluginsdata );
		sort( $pluginkeys );
		// print_r($pluginkeys);

		// --- set plugin directory ---
		// 0.9.2: allow for plugin root directory override
		// 0.9.3: use already set plugin root directory
		$plugin_root = $paramedic['plugindir'];

		// --- set plugin data and inactive plugin array ---
		$paramedic['plugins']['inactive'] = array();
		foreach ( $pluginkeys as $pluginkey ) {
			$plugindata = get_plugin_data( $plugin_root . $pluginkey );
			$paramedic['plugins']['data'][$pluginkey] = $plugindata;
			if ( !in_array( $pluginkey, $paramedic['plugins']['active'] ) ) {
				$paramedic['plugins']['inactive'][] = $pluginkey;
			}
		}

		return $paramedic['plugins'];
	}
}

// -------------------------
// Set All Theme Global Data
// -------------------------
if ( !function_exists( 'paramedic_get_theme_data' ) ) {
	function paramedic_get_theme_data() {

		global $paramedic;

		$paramedic['themes']['test'] = '';

		// --- get theme data ---
		if ( !function_exists('wp_get_themes' ) ) {
			include_once ABPATH . WPINC . '/theme.php';
		}

		// --- set theme root ---
		// 0.9.4: use already set theme root directory
		// $theme_root = get_theme_root();
		$theme_root = $paramedic['themedir'];
		register_theme_directory( $theme_root );
		$paramedic['themes']['data'] = wp_get_themes();

		// --- set inactive theme list ---
		$paramedic['themes']['inactive'] = array();
		foreach ( $paramedic['themes']['data'] as $theme => $data ) {
			if ( $theme != $paramedic['themes']['active'] ) {
				$paramedic['themes']['inactive'][] = $theme;
			}
		}

		// --- set network themes ---
		// 0.9.4: get active themes for all blogs on multisite
		if ( is_multisite() ) {
			$sites = get_sites( array( 'fields' => 'ids' ) );
			foreach ( $sites as $site_id ) {
				$template = get_blog_option( $site_id, 'template' );
				if ( !isset($paramedic['themes']['network']) || !in_array( $template, $paramedic['themes']['network'] ) ) {
					$paramedic['themes']['network'][] = $template;
				}
				$stylesheet = get_blog_option( $site_id, 'stylesheet' );
				if ( !isset($paramedic['themes']['network']) || !in_array( $stylesheet, $paramedic['themes']['network'] ) ) {
					$paramedic['themes']['network'][] = $stylesheet;
				}
			}
		}

		return $paramedic['themes'];
	}
}

// ---------------------
// Filter Active Plugins
// ---------------------
// ...where the magic happens...
// ('active_plugins' option)
if ( !function_exists( 'paramedic_active_plugins_filter' ) ) {

	add_filter( 'option_active_plugins', 'paramedic_active_plugins_filter', 0 );

	function paramedic_active_plugins_filter( $plugins ) {

		global $paramedic;

		// --- check function trace ---
		// (make sure this filter is called via wp_get_active_and_valid_plugins in wp-settings.php)
		$check = paramedic_match_trace( 'wp_get_active_and_valid_plugins', 'wp-settings.php', 10 );
		if ( !$check ) {return $plugins;}

		// --- test for a pluggable function ---
		// (as these are loaded directly after plugins, presence indicates plugins already loaded)
		// if ( function_exists( 'wp_set_current_user' ) ) {return $plugins;}

		if ( isset( $paramedic['test'] ) && $paramedic['test'] ) {
			if ( isset( $paramedic['testtype'] ) && ( 'plugin' == $paramedic['testtype'] ) ) {
				if ( count( $paramedic['plugins']['test'] ) > 0 ) {

					// --- remove filter later for proper plugins.php page load ---
					if ( is_admin() ) {paramedic_plugins_page_check();}

				    // --- remove plugins being inverse tested ---
				    // 0.9.1: handle inverse plugins test
				    if ( $paramedic['plugins']['inversetest'] ) {
				    	foreach ( $plugins as $i => $plugin ) {
							if ( in_array( $plugin, $paramedic['plugins']['test'] ) ) {
								unset( $plugins[$i] );
							}
						 }
				    }
					$paramedic['plugins']['load'] = $plugins;
				 } else {
					 $paramedic['plugins']['load'] = $paramedic['plugins']['test'];
				 }

				 // --- make sure we do not attempt to load plugins twice ---
				 if ( isset( $paramedic['plugins']['loaded'] ) && $paramedic['plugins']['loaded'] ) {
					 return $paramedic['plugins']['load'];
				 }

				 // --- actually load plugins early! ---
				 // 0.9.4: load plugins here to test for output
				 // (code via wp-includes/load.php)
				 $network_plugins = is_multisite() ? wp_get_active_network_plugins() : false;
				 foreach ( $paramedic['plugins']['load'] as $plugin ) {
					 $pluginpath = WP_PLUGIN_DIR . '/' . $plugin;
					 if ( !validate_file( $plugin ) && ( '.php' == substr( $plugin, - 4 ) ) && file_exists( $pluginpath )
					      && ( !$network_plugins || !in_array( WP_PLUGIN_DIR . '/' . $plugin, $network_plugins ) ) ) {

						 // --- sandbox load the plugin to detect output ---
						 wp_register_plugin_realpath( $pluginpath );
						 ob_start();
						 paramedic_stats_start( $plugin );

						 // 0.9.5: fix to preserve plugin variable value
						 // (as plugin is a too common variable name)
						 $unique_plugin_variable = $plugin;
						 include_once $pluginpath;
						 $plugin = $unique_plugin_variable;
						 paramedic_stats_end( $plugin );
						 $output = ob_get_contents();
						 ob_end_clean();

						 // --- store the unexpected output if any ---
						 if ( is_string( $output ) && ( strlen( $output ) > 0 ) ) {
							 $paramedic['pluginoutput'][$plugin] = $output;
						 }
					 }
				 }

				 // --- set plugins loaded switch to be safe ---
				 $paramedic['plugins']['loaded'] = true;

				 // --- for standard plugins test ---
				 // 0.9.4: return empty as already loaded
				 // return $paramedic['plugins']['test'];
				 return array();
			}

			// --- load no plugins for core and theme tests ---
		    return array();
		}
		return $plugins;
	}
}

// ------------------------
// Admin Plugins Page Check
// ------------------------
// remove the filter for wp-admin/plugins.php to prevent it updating active_plugins
// (in an edge case testing inactive plugins that somehow fail validation)
if ( !function_exists( 'paramedic_plugins_page_check' ) ) {
	function paramedic_plugins_page_check() {
		global $pagenow;
		if ( 'plugins.php' == $pagenow ) {
			if ( !has_action( 'load-plugins.php', 'paramedic_remove_plugins_filter' ) ) {
				add_action( 'load-plugins.php', 'paramedic_remove_plugins_filter' );
			}
		}
	}
}

// ----------------------------
// Remove Active Plugins Filter
// ----------------------------
 if ( !function_exists( 'paramedic_remove_plugins_filter' ) ) {
	 function paramedic_remove_plugins_filter() {
		 remove_filter( 'option_active_plugins', 'paramedic_active_plugins_filter', 0 );
	 }
 }

// ----------------------------------
// Set Must Use Plugins Loaded Switch
// ----------------------------------
 if ( !function_exists( 'paramedic_muplugins_loaded' ) ) {
	 add_action( 'muplugins_loaded', 'paramedic_muplugins_loaded' );
	 add_action( 'muplugins_loaded_multisite', 'paramedic_muplugins_loaded' );
	 function paramedic_muplugins_loaded() {
		 global $paramedic;
		 $paramedic['plugins']['muloaded'] = true;
	 }
 }

// -----------------------
// Filter Theme Stylesheet
// -----------------------
if (!function_exists('paramedic_stylesheet_filter')) {

	add_filter('option_stylesheet', 'paramedic_stylesheet_filter' );

	function paramedic_stylesheet_filter ($theme ) {

		global $paramedic;

		if ( isset( $paramedic['test'] ) && $paramedic['test'] ) {
			if ( isset( $paramedic['testtype'] ) ) {

				// --- return theme to test if doing a theme test ---
				if ( ( 'theme' == $paramedic['testtype'] ) && isset( $paramedic['themes']['test'] ) ) {
					return $paramedic['themes']['test'];
				}

				// --- maybe return theme override if doing a plugin test ---
				if ( ( 'plugin' == $paramedic['testtype'] ) && isset( $paramedic['usetheme'] ) ) {
					return $paramedic['usetheme'];
				}
			}

			// TODO: option to maybe fallback to latest "default" theme for plugin testing ?
			// eg. return 'twentyten';
			// foreach ( $paramedic['themes']['data'] as $themeslug => $data ) {
			//	$defaultthemes = array( 'twenty', 'nineteen', 'eighteen', 'seventeen', 'sixteen', 'fifteen', 'fourteen', 'thirteen', 'twelve', 'eleven', 'ten' );
			//	foreach ( $defaultthemes as $i => $slug ) {
			//		if ( 'twenty' . $slug == $themeslug ) {$themeslugs[$i] = $themeslug;}
			//	}
			//	if ( count( $themeslugs ) > 0 ) {
			//      $lowestkey = '???';
			//      return $themeslugs[$lowestkey];
			//  }
			// }
		}
		return $theme;
	}
}

// ---------------------
// Filter Theme Template
// ---------------------
// (for child themes)
if ( !function_exists( 'paramedic_template_filter' ) ) {

	add_filter( 'option_template', 'paramedic_template_filter' );

	function paramedic_template_filter( $template ) {

		global $paramedic;

		if ( isset( $paramedic['test'] ) && $paramedic['test'] ) {
			if ( isset( $paramedic['testtype'] ) ) {

				// --- return theme template to test if doing a theme test ---
				if ( 'theme' == $paramedic['testtype'] ) {
					$testtheme = $paramedic['themes']['test'];
					$theme = wp_get_theme( $testtheme );
					return $theme['Template'];
				}

				// --- maybe return theme template override if doing a plugin test ---
				if ( ( 'plugin' == $paramedic['testtype'] ) && isset( $paramedic['usetheme'] ) ) {
					$theme = wp_get_theme( $paramedic['usetheme'] );
					return $theme['Template'];
				}
			}
		}
		return $template;
	}
}

// -----------------------------
// Filter Active Network Plugins
// -----------------------------
// ('active_sitewide_plugins' on multisite)
if ( !function_exists('paramedic_network_plugins_filter' ) ) {

	add_filter( 'site_option_active_sitewide_plugins', 'paramedic_network_plugins_filter', 10, 3 );

	function paramedic_network_plugins_filter( $plugins, $option, $network_id ) {

		global $paramedic;

		// TODO: better handle $network_id by comparing to get_current_network_id()?

		// --- check function trace ---
		// (make sure this filter is called via wp_get_active_network_plugins in wp-settings.php)
		$check = paramedic_match_trace( 'wp_get_active_network_plugins', 'wp-settings.php', 10 );
		if ( !$check ) {
			return $plugins;
		}

		// --- where mu-plugins have really finished loading on multisite ---
		// (before sitewide network plugins are loaded)
		do_action( 'muplugins_loaded_multisite' );

		// --- check if doing network plugin test ---
		if ( isset($paramedic['test']) && $paramedic['test'] ) {
		    if ( ( isset($paramedic['testtype'] ) ) && ('network' == $paramedic['testtype'] ) ) {

				// --- return plugins to test if doing a network plugin test ---
				if ( ( count( $paramedic['plugins']['test'] ) > 0 ) && ( count( $plugins ) > 0 ) ) {
					$plugins = array_intersect( $paramedic['plugins']['test'], array_keys( $plugins ) );
				}

				// --- sort active plugins ---
				$active_plugins = array_keys( $plugins );
				sort( $active_plugins );

				// --- store network load plugins ---
				$paramedic['plugins']['networkload'] = $active_plugins;

				// --- check network plugins loaded switch ---
				if ( isset( $paramedic['plugins']['networkloaded'] ) && $paramedic['plugins']['networkloaded'] ) {
					return $active_plugins;
				}

				// --- load sitewide network plugins now! ---
				// (code via ms-load.php)
				// 0.9.4: load early to check for unexpected output
				foreach ( $active_plugins as $plugin ) {
					$pluginpath = WP_PLUGIN_DIR . '/' . $plugin;
					if ( !validate_file( $plugin ) && ( '.php' == substr($plugin, -4 ) ) && file_exists( $pluginpath ) ) {
		                ob_start();
		                paramedic_stats_start( $plugin );
						include_once $pluginpath;
						paramedic_stats_end( $plugin );
						$output = ob_get_contents();
						ob_end_clean();
						if ( is_string( $output ) && ( strlen( $output) > 0 ) ) {
							$paramedic['pluginoutput'][$plugin] = $output;
						}
					}
				}

				// --- set network plugins loaded switch ---
				$paramedic['plugins']['networkloaded'] = true;

				// --- return empty as already loaded ---
				return array();

			} else {
				// --- for a core/plugin/theme test load no network plugins ---
				return array();
			}
		}

		return $plugins;
	}
}

// --------------------
// Get User Auth Cookie
// --------------------
if ( !function_exists( 'paramedic_get_auth_cookies' ) ) {
	function paramedic_get_auth_cookies( $userid ) {

		global $paramedic;

		// --- filter to not send cookies to browser ---
		add_filter( 'send_auth_cookies', '__return_false', 11 );

		// --- filter expiration time (short) ---
		add_filter( 'auth_cookie_expiration', 'paramedic_auth_expiry', 11, 3 );

		// --- add actions to capture cookies ---
		add_action( 'set_auth_cookie', 'paramedic_capture_auth_cookie', 11, 6 );
		add_action( 'set_logged_in_cookie', 'paramedic_capture_login_cookie', 11, 6 );

		// --- call set auth cookie to generate cookies ---
		wp_set_auth_cookie( $userid, false, is_ssl() );

		// --- remove filters and actions ---
		remove_filter( 'send_auth_cookies', '__return_false', 11 );
		remove_filter( 'auth_cookie_expiration', 'paramedic_auth_expiry', 11, 3 );
		remove_action( 'set_auth_cookie', 'paramedic_capture_auth_cookie', 11, 6 );
		remove_action( 'set_logged_in_cookie', 'paramedic_capture_login_cookie', 11, 6 );

		// --- debug point ---
		// echo "<!-- User " . $userid . " Auth Cookies: " . print_r( $cookies, true ) . " -->";

		return $cookies;
	}
}

// -------------------------------
// Temporary Authentication Expiry
// -------------------------------
// 0.9.4: set temp auth cookie expiry to match test time limit
if ( !function_exists( 'paramedic_auth_expiry' ) ) {
	function paramedic_auth_expiry( $time, $userid, $remember ) {
		global $paramedic;
		$time = time() + $paramedic['timelimit'];
		return $time;
	}
}

// -------------------
// Capture Auth Cookie
// -------------------
// 0.9.4: capture the user auth cookie
// note: how the auth cookie is normally set
// setcookie($auth_cookie_name, $auth_cookie, $expire, PLUGINS_COOKIE_PATH, COOKIE_DOMAIN, $secure, true);
// setcookie($auth_cookie_name, $auth_cookie, $expire, ADMIN_COOKIE_PATH, COOKIE_DOMAIN, $secure, true);
if ( !function_exists( 'paramedic_capture_auth_cookie' ) ) {
	function paramedic_capture_auth_cookie( $auth_cookie, $expire, $expiration, $user_id, $scheme, $token ) {
		global $paramedic;
		if ( 'secure_auth' == $scheme ) {
			$cookie_name = SECURE_AUTH_COOKIE;
		} else {
			$cookie_name = AUTH_COOKIE;
		}
		$paramedic['cookies'][$cookie_name] = $auth_cookie;
	}
}

// --------------------
// Capture Login Cookie
// --------------------
// 0.9.4: capture the user login cookie
// note: how the logged in cookie is normally set
// setcookie(LOGGED_IN_COOKIE, $logged_in_cookie, $expire, COOKIEPATH, COOKIE_DOMAIN, $secure_logged_in_cookie, true);
// if ( COOKIEPATH != SITECOOKIEPATH )
// 	setcookie(LOGGED_IN_COOKIE, $logged_in_cookie, $expire, SITECOOKIEPATH, COOKIE_DOMAIN, $secure_logged_in_cookie, true);
if ( !function_exists( 'paramedic_capture_login_cookie' ) ) {
	function paramedic_capture_login_cookie( $logged_in_cookie, $expire, $expiration, $user_id, $scheme, $token ) {
		global $paramedic;
		// if ( COOKIEPATH == SITECOOKIEPATH ) {$path = COOKIEPATH;} else {$path = SITECOOKIEPATH;}
		$paramedic['cookies'][LOGGED_IN_COOKIE] = $logged_in_cookie;
	}
}

// --------------------
// Get Plugins Function
// --------------------
// standalone get_plugins function copy - minus caching!
if ( !function_exists('paramedic_get_plugins' ) ) {
	function paramedic_get_plugins( $plugin_folder = '' ) {

		// --- maybe include admin plugin.php file ---
		if ( !function_exists( 'get_plugin_data' ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		// [caching disabled]
		// if ( ! $cache_plugins = wp_cache_get('plugins', 'plugins') )
		//	$cache_plugins = array();
		// if ( isset($cache_plugins[ $plugin_folder ]) )
		// 	return $cache_plugins[ $plugin_folder ];

		$wp_plugins = array();

		// --- set plugins root ---
		// 0.9.2: allow for plugin root directory override
		// 0.9.3: use already set plugin root directory
		global $paramedic;
		$plugin_root = $paramedic['plugindir'];

		if ( !empty( $plugin_folder ) ) {
			$plugin_root .= $plugin_folder;
		}

		// --- Files in wp-content/plugins directory ---
		$plugins_dir = @opendir( $plugin_root );
		$plugin_files = array();
		if ( $plugins_dir ) {
			while ( ( $file = readdir( $plugins_dir ) ) !== false ) {
				if ( substr( $file, 0, 1 ) == '.' ) {
					continue;
				}
				if ( is_dir( $plugin_root . '/' . $file ) ) {
					$plugins_subdir = @opendir( $plugin_root . '/' . $file );
					if ( $plugins_subdir ) {
						while ( ( $subfile = readdir( $plugins_subdir ) ) !== false ) {
							if ( substr( $subfile, 0, 1 ) == '.' ) {
								continue;
							}
							if ( substr( $subfile, - 4 ) == '.php' ) {
								$plugin_files[] = $file . '/' . $subfile;
							}
						}
						closedir( $plugins_subdir );
					}
				} elseif ( substr( $file, - 4 ) == '.php' ) {
					$plugin_files[] = $file;
				}
			}
			closedir( $plugins_dir );
		}

		if ( empty( $plugin_files ) ) {
			return $wp_plugins;
		}

		foreach ( $plugin_files as $plugin_file ) {
			if (!is_readable( $plugin_root.'/' . $plugin_file ) ) {
				continue;
			}
			$plugin_data = get_plugin_data( $plugin_root . '/' . $plugin_file, false, false );
			if ( empty( $plugin_data['Name'] ) ) {
				continue;
			}
			$wp_plugins[plugin_basename( $plugin_file )] = $plugin_data;
		}

		uasort( $wp_plugins, '_sort_uname_callback' );

		// [caching disabled]
		// $cache_plugins[ $plugin_folder ] = $wp_plugins;
		// wp_cache_set('plugins', $cache_plugins, 'plugins');

		return $wp_plugins;
	}
}

// -------------------
// Get Themes Function
// -------------------
// [unused] (this is a copy of wp_get_themes function)
if (!function_exists('paramedic_get_all_themes')) {
	function paramedic_get_all_themes( $args = array() ) {

		global $wp_theme_directories;

		$defaults = array( 'errors' => false, 'allowed' => null, 'blog_id' => 0 );
		$args = wp_parse_args( $args, $defaults );

		$theme_directories = search_theme_directories();
		// print_r( $theme_directories );

		if ( count( $wp_theme_directories ) > 1 ) {
			// Make sure the current theme wins out, in case search_theme_directories() picks the wrong
			// one in the case of a conflict. (Normally, last registered theme root wins.)
			$current_theme = get_stylesheet();
			if ( isset( $theme_directories[ $current_theme ] ) ) {
				$root_of_current_theme = get_raw_theme_root( $current_theme );
				if ( ! in_array( $root_of_current_theme, $wp_theme_directories ) ) {
					$root_of_current_theme = WP_CONTENT_DIR . $root_of_current_theme;
				}
				$theme_directories[ $current_theme ]['theme_root'] = $root_of_current_theme;
			}
		}

		if ( empty( $theme_directories ) ) {
			return array();
		}

		if ( is_multisite() && null !== $args['allowed'] ) {
			$allowed = $args['allowed'];
			if ( 'network' === $allowed ) {
				$theme_directories = array_intersect_key( $theme_directories, WP_Theme::get_allowed_on_network() );
			} elseif ( 'site' === $allowed ) {
				$theme_directories = array_intersect_key( $theme_directories, WP_Theme::get_allowed_on_site( $args['blog_id'] ) );
			} elseif ( $allowed ) {
				$theme_directories = array_intersect_key( $theme_directories, WP_Theme::get_allowed( $args['blog_id'] ) );
			} else {
				$theme_directories = array_diff_key( $theme_directories, WP_Theme::get_allowed( $args['blog_id'] ) );
			}
		}

		$themes = array();
		static $_themes = array();

		foreach ( $theme_directories as $theme => $theme_root ) {
			if ( isset( $_themes[ $theme_root['theme_root'] . '/' . $theme ] ) ) {
				$themes[ $theme ] = $_themes[ $theme_root['theme_root'] . '/' . $theme ];
			} else {
				$themes[ $theme ] = $_themes[ $theme_root['theme_root'] . '/' . $theme ] = new WP_Theme( $theme, $theme_root['theme_root'] );
			}
		}

		if ( null !== $args['errors'] ) {
			foreach ( $themes as $theme => $wp_theme ) {
				if ( $wp_theme->errors() != $args['errors'] ) {
					unset( $themes[ $theme ] );
				}
			}
		}

		return $themes;
	}
}


// ----------------------------
// === Handle Test Requests ===
// ----------------------------

// ----------------
// Test Finish Exit
// ----------------
// (used for exiting before theme_setup)
if ( !function_exists( 'paramedic_test_finish' ) ) {
	function paramedic_test_finish() {
		global $paramedic;

		// --- record test stats ---
		// 0.9.5: add load stats on test shutdown
		if ( !isset($paramedic['inversetest'] ) || !$paramedic['inversetest'] ) {

		    $memory = memory_get_usage() - $paramedic['startmemory'];
		    $loadtime = microtime( true ) - $paramedic['starttime'];
		    $queries = get_num_queries() - $paramedic['startqueries'];
			// 0.9.5: fix for fatal error edge case where resources key is not set
			if ( !isset( $paramedic['resources'] ) ) {
				$paramedic['resources'] = 'UNKNOWN';
			}
		    $stats = '[*' . $paramedic['resources'] . '*] M:' . $memory . ',T:' . $loadtime . ',Q:' . $queries . PHP_EOL;
		    error_log( $stats, 3, $paramedic['logpath'] );

		} elseif ( isset( $paramedic['inversetest'] ) && $paramedic['inversetest'] ) {

		    // TODO: maybe handle inverse test statistics also?

		}

		exit;
	}
}

// ------------------------------------
// Check and Verify Single Test Request
// ------------------------------------
if ( isset( $_REQUEST['testtoken'] ) && isset( $_REQUEST['testid'] ) ) {

	// --- get test token for test ID ---
	$token = get_transient( 'paramedic_token_' . $paramedic['testid'] );

	// --- check test token against request ---
	if ( $token && ( $token == trim( $_REQUEST['testtoken'] ) ) ) {

		// --- set empty resource array ---
		$resources = array();

		// --- define error test flag ---
		$paramedic['test'] = true;

        // --- set test start stats ---
        // 0.9.5: added for stats calculation
        $paramedic['startmemory'] = memory_get_usage();
        $paramedic['starttime'] = microtime( true );
        $paramedic['startqueries'] = get_num_queries();

		// --- populate theme and plugin data ---
		$paramedic['themes'] = paramedic_get_theme_data();
		$paramedic['plugins'] = paramedic_get_plugin_data();

		// --- handle core test request ---
		if ( isset( $_REQUEST['testcore'] ) ) {
			$coretest = trim( strtolower( $_REQUEST['testcore'] ) );
			if ( 'yes' == $coretest ) {
				// --- filter out to load no plugins ---
				$paramedic['testtype'] = 'core';
				$resources[] = 'core';

				// --- set to finish before theme is loaded (setup_theme action) ---
				// TODO: maybe need to exit before plugins are loaded ?
				add_action( 'setup_theme', 'paramedic_test_finish', 0 );
			}
		}

		// --- maybe set plugins test array ---
		if ( isset( $_REQUEST['testplugin'] ) ) {
			$testplugin = trim( strtolower( $_REQUEST['testplugin'] ) );
			if ( '' != $testplugin ) {
				$paramedic['testtype'] = 'plugin';
				if ( strstr( ',', $testplugin ) ) {
					$testplugins = explode( ',', $testplugin );
				} else {
					$testplugins[0] = $testplugin;
				}

				foreach ($testplugins as $testplugin) {
					$testplugin = trim( $testplugin );
					if ( array_key_exists( $testplugin, $paramedic['plugins']['data'] ) ) {
						$paramedic['plugins']['test'][] = $testplugin;
						$resources[] = $testplugin;
					}
				}

                // --- collect stats on shutdown ---
                // 0.9.5: added for stats collection
                add_action( 'shutdown', 'paramedic_test_finish' );
            }
		}

		// --- check for inverse plugin test ---
		// 0.9.1: maybe set switch for an inverse plugin test
		// 0.9.3: simplify logic for test inverse switch check
		// 0.9.6: fix to incorrect variable name
		if ( isset( $_REQUEST['inversetest'] ) ) {
			$testinverse = $_REQUEST['inversetest'];
		} else {
			$testinverse = false;
		}
		if ( ( '1' == $testinverse ) || ( 'yes' == $testinverse ) ) {
			$testinverse = true;
		} else {
			$testinverse = false;
		}
		$paramedic['plugins']['inversetest'] = $testinverse;

		// --- maybe set theme override (to test plugins with) ---
		if ( isset( $_REQUEST['usetheme'] ) ) {
			$usetheme = trim( strtolower( $_REQUEST['usetheme'] ) );
			if ( 'none' == $usetheme ) {
				// --- exit early if no theme load test is required ---
				add_action( 'setup_theme', 'paramedic_test_finish', 0 );
			} else {
				// --- set theme override and add to resource list ---
                $resources[] = $paramedic['usetheme'] = $usetheme;
            }
        }

   		// --- maybe set theme test string ---
		if ( isset( $_REQUEST['testtheme'] ) ) {
			$testtheme = trim( strtolower( $_REQUEST['testtheme'] ) );
			// TODO: maybe allow for multiple theme testing ?
			if ( '' != $testtheme ) {
				if ( array_key_exists( $testtheme, $paramedic['themes']['data'] ) ) {
					$paramedic['testtype'] = 'theme';
					$paramedic['themes']['test'] = $testtheme;
                    $resources[] = $testtheme;

                    // --- collect stats on shutdown ---
                    // 0.9.5: added for stats collection
                    add_action( 'shutdown', 'paramedic_test_finish' );
				}
			}
		}

		// --- if we have resources then the test is going ahead ---
		if ( count( $resources ) > 0 ) {

			// --- set resources list ---
			// (may be a combination of theme/plugin slugs)
			$paramedic['resources'] = implode( ',', $resources );

			// --- set debug log and error logging PHP settings ---
            // (note: this overrides WP_DEBUG settings for test pageload)
            // 0.9.5: disable error logging as we are now catching errors
			ini_set( 'error_log', $paramedic['logpath'] );
			ini_set( 'log_errors', 0 );
			error_reporting( -1 );

			// ---- initialize test error handler class ---
			global $ParamedicErrorHandler;
			$ParamedicErrorHandler = new ParamedicErrorHandler();
        }

	}
}


// ---------------------------
// === Check Test Triggers ===
// ---------------------------
// 0.9.4: check for paramedic querystring trigger
if ( isset( $_REQUEST['paramedic'] ) ) {

	// --- set test trigger keywords ---
	// (possible querystring trigger values)
	$testtriggers = array( 'active', 'yes', '1' );

	// --- so we know whether to exit on completion of tests ---
	$testcore = $testmuplugins = $testplugins = $testthemes = $donetests = false;

	// --- set empty error arrays for passing between tests ---
	$coreerrors = $themeerrors = $pluginerrors = array();

	// maybe set test user
	// -------------------
	// 0.9.4: allow setting of current / guest / user ID
	// note: no cookies = not logged in = guest)
	$userlogin = false;
	$paramedic['user'] = 'current';
	$valid = array( 'current', 'guest', 'userid' );
	if ( defined( 'PARAMEDIC_USER' ) && PARAMEDIC_USER ) {
		$valid = array( 'current', 'guest' );
		if ( in_array( PARAMEDIC_USER, $valid ) ) {
			$paramedic['user'] = PARAMEDIC_USER;
		} elseif ( is_numeric( PARAMEDIC_USER ) ) {
			$paramedic['user'] = PARAMEDIC_USER;
		} else {
			$userlogin = sanitize_user( PARAMEDIC_USER );
		}
	} elseif ( isset( $_REQUEST['user'] ) && in_array( $_REQUEST['user'], $valid ) ) {
		if ( $_REQUEST['user'] == 'userid' ) {
			paramedic_recheck_user_auth();
			if ( $paramedic['authed'] && isset( $_REQUEST['userid'] ) ) {
				if (is_numeric( $_REQUEST['userid'] ) ) {
					$paramedic['user'] = $_REQUEST['userid'];
				} else {
					$userlogin = sanitize_user( $_REQUEST['userid'] );
				}
			} else {
				// --- not authed error message ---
				echo esc_html( __( 'Security key required to set test user.', 'paramedic' ) );
				exit;
			}
		} else {
			$paramedic['user'] = $_REQUEST['user'];
		}
	}
	if ( $userlogin ) {
		$user = get_user_by( 'login', $userlogin );
		unset( $userlogin );
		if ( $user ) {
			$paramedic['user'] = $user->ID;
			unset( $user );
		}
	}
	if ( is_numeric( $paramedic['user'] ) ) {
		paramedic_recheck_user_auth();
		if ( is_user_logged_in() && ( $paramedic['user'] == get_current_user_id() ) ) {
			$paramedic['user'] = 'current';
		}
		if ( ( 0 === $paramedic['user'] ) || ( '0' == $paramedic['user'] ) ) {
			$paramedic['user'] = 'guest';
		}
	}

	// maybe log test results separately
	// ---------------------------------
	$paramedic['testlog'] = false;
	if ( ( defined( 'PARAMEDIC_TESTLOG' ) && PARAMEDIC_TESTLOG )
	  || ( isset( $_REQUEST['testlog'] ) && in_array( strtolower( $_REQUEST['testlog']), $testtriggers ) ) ) {
		$paramedic['testlog'] = true;
	}

	// check for silent switch
	// -----------------------
	// 0.9.4: set settings key value directly
	$paramedic['silent'] = false;
	if ( defined( 'PARAMEDIC_SILENT' ) ) {
		$paramedic['silent'] = PARAMEDIC_SILENT;
	} elseif ( isset($_REQUEST['silent'] ) ) {
		if ( ( 'yes' == strtolower( $_REQUEST['silent'] ) ) || ( '1' == $_REQUEST['silent'] ) ) {
			$paramedic['silent'] = true;
		}
	}

	// check trigger for core test
	// ---------------------------
	// note: loads no plugins or theme, but currently includes mu-plugins/network active plugin loading
	if ( isset($_REQUEST['coretest'] ) ) {
		$testcore = trim( strtolower( $_REQUEST['coretest'] ) );
		if (in_array( $testcore, $testtriggers ) ) {
			$testcore = true;
		} else {
			$testcore = false;
		}
	}

	// check trigger for must-use plugins test
	// ---------------------------------------
	// TODO: add test method for mu-plugins
	// if ( isset($_REQUEST['mutest'] ) ) {
	// 	$mutest = trim( strtolower( $_REQUEST['muplugintest'] ) );
	//	if ( in_array( $muplugintest, $testtriggers ) ) {$testmuplugins = true;}
	// }

	// check trigger for plugin test(s)
	// --------------------------------
	// for plugin tests, core is tested to first separate out errors
	if ( isset( $_REQUEST['plugintest'] ) ) {
		$testplugins = trim( strtolower( sanitize_text_Field( $_REQUEST['plugintest'] ) ) );
		if ( in_array( $testplugins, $testtriggers ) ) {
			$testcore = true;
			$testplugins = 'active';
		} elseif ( ( 'inactive' == $testplugins ) || ( 'all' == $testplugins ) ) {
			$testcore = true;
		} elseif ( is_multisite() && ( 'network' == $testplugins ) ) {
			$testcore = true;
		} else {
			// $testplugins = false;
		}

		// --- check for theme override (to test plugins with) ---
		// if ( isset( $_REQUEST['usetheme'] ) ) {
		// 	$usetheme = trim( strtolower( $_REQUEST['usetheme'] ) );
		//	if ( ( '' != $usetheme ) && ( array_key_exists( $usetheme, $paramedic['themes']['data'] ) ) ) {
		//		$testthemes = $paramedic['usetheme'] = $usetheme;
		//	}
		// }
	}

	// check trigger for theme test(s)
	// -------------------------------
	// for theme tests, plugins are tested first to separate out errors
	if ( isset( $_REQUEST['themetest'] ) ) {
		$testthemes = trim( strtolower( sanitize_text_field( $_REQUEST['themetest'] ) ) );
		if ( in_array( $testthemes, $testtriggers ) ) {
			$testthemes = 'active';
			$testcore = true;
		} elseif ( ( 'inactive' == $testthemes ) || ( 'all' == $testthemes ) ) {
			$testcore = true;
		} else {
			$testthemes = false;
		}
	}

	// check trigger for all tests
	// ---------------------------
	if ( isset( $_REQUEST['testall'] ) ) {
		$testall = trim( strtolower( sanitize_text_field( $_REQUEST['testall'] ) ) );
		$testcore = true;
		if ( ( 'yes' == $testall ) || ( '1' == $testall ) ) {
			// 0.9.4: set test inverse to new 'all' value
			$testthemes = $testplugins = $testinverse = 'all';
		} elseif ( 'active' == $testall ) {
			$testthemes = $testplugins = 'active';
		} elseif ( 'inactive' == $testall ) {
			$testthemes = $testplugins = 'inactive';
		} elseif ( is_multisite() && ( 'network' == $testall ) ) {
			// 0.9.4: test active themes for blog network
			$testthemes = $testplugins = 'network';
		}
	}

	// check trigger for inverse (plugin) test
	// ---------------------------------------
	// 0.9.1: added inverse test querystring overrides
	// 0.9.3: set testinverse default to prevent undefined index warning
	// 0.9.4: set default to 'yes' and add new possible 'all' value
	if ( !isset( $testinverse ) ) {
		$testinverse = 'yes';
	}
	if ( defined( 'PARAMEDIC_INVERSE' ) ) {
		$testinverse = PARAMEDIC_INVERSE;
	} elseif ( isset($_REQUEST['inverse'] ) ) {
		$testinverse = sanitize_text_field( $_REQUEST['inverse'] );
	}
	if ( ( '0' == $testinverse ) || ( 'no' == $testinverse ) ) {
		$testinverse = 'no';
	} elseif ( ( '1' == $testinverse ) || ( 'yes' == $testinverse ) ) {
		$testinverse = 'yes';
	} elseif ( 'all' == $testinverse ) {
		$testinverse = 'all';
	}
	$paramedic['plugins']['testinverse'] = $testinverse;

	// --- maybe output current test debug info ---
    // 0.9.4: echo test parameters if debugging
    // 0.9.5: fix to == typo in variable setting
    $coremessage = $mumessage = $pluginmessage = $thememessage = 'no';
	if ( $testcore ) {
		$coremessage = 'yes';
	}
	// if ( $testmuplugins ) {$mumessage = 'yes';}
	if ( $testplugins ) {
		$pluginmessage = $testplugins;
	}
	if ( $testthemes ) {
		$thememessage = $testthemes;
	}
	paramedic_debug(
		"Test ID: " . $paramedic['testid'] . PHP_EOL .
		"User: " . $paramedic['user'] . PHP_EOL .
		"Core Test: " . $coremessage . PHP_EOL .
		// "Must Use Plugins Test: " . $mumessage . PHP_EOL .
		"Plugins Test: " . $pluginmessage . PHP_EOL .
		"Theme Test: " . $thememessage . PHP_EOL .
		"Test Inverse: " . $paramedic['plugins']['testinverse'] . PHP_EOL
	);
	if ($paramedic['authed']) {
		paramedic_debug(
		 	"Core Dir: " . ABSPATH . PHP_EOL .
			"Debug Log Path: " . $paramedic['logpath']. PHP_EOL .
			"Theme Dir: " . $paramedic['themedir'] . PHP_EOL .
			"Plugin Dir: " . $paramedic['themedir'] . PHP_EOL
		);
	}


	// -------------------
	// === Start Tests ===
	// -------------------

	$paramedic['interface'] = false;

    // maybe do core test
	// ------------------
	// includes any mu-plugins (but not network activated plugins in multisite)
	if ( $testcore ) {
		$coretest = paramedic_do_test( 'core', false );
		if ( $coretest && is_array( $coretest ) ) {
			$donetests = true;
			$coreerrors = paramedic_collate_errors( $coretest, 'core' );
			if ( !$paramedic['silent'] ) {
				paramedic_interface();
				echo '<div id="core-errors">' . PHP_EOL;
			}
			$results = paramedic_output_results( $coreerrors, $coretest, 'core' );
			if ( !$paramedic['silent'] ) {
				echo '</div><!-- /#core-errors -->' . PHP_EOL;
			}
			if ( $results != '' ) {
				paramedic_log_results( $results );
			}
		}
	}

	// maybe do plugin(s) test
	// -----------------------
	if ( $testplugins ) {

		if ( ( 'network' == $testplugins ) && !is_multisite() ) {
			// do nothing as there are no network activated plugins without multisite!
		} else {
			$plugintest = paramedic_do_test( 'plugins', $testplugins );
		}

		if ( isset( $plugintest ) && $plugintest && is_array( $plugintest ) ) {
			$donetests = true;
			$errors = array_merge( $coreerrors, $themeerrors );
			$pluginerrors = paramedic_collate_errors( $plugintest, 'plugins', $errors );
			if ( !$paramedic['silent'] ) {
				paramedic_interface();
				echo '<div id="plugin-errors">' . PHP_EOL;
			}
			$results = paramedic_output_results( $pluginerrors, $plugintest, 'plugin' );
			if ( !$paramedic['silent'] ) {
				echo '</div><!-- /#plugin-errors -->' . PHP_EOL;
			}
			if ( '' != $results ) {
				paramedic_log_results( $results );
			}
		} elseif ( isset( $plugintest ) && !$plugintest ) {
			echo '<div style="font-size:18px; font-family: Consolas, \'Lucida Console\', Monaco, FreeMono, monospace">';
			echo '<b>' . esc_html( __( 'No matching Plugins found to test.', 'paramedic' ) ) . '</b></font><br><br>';
		}
	}

	// maybe do theme(s) test
	// ----------------------
	if ( $testthemes ) {
		$themetest = paramedic_do_test( 'themes', $testthemes );
		if ( $themetest && is_array( $themetest ) ) {
			$donetests = true;
			$themeerrors = paramedic_collate_errors( $themetest, 'theme', $coreerrors );
			if ( !$paramedic['silent'] ) {
				paramedic_interface();
				echo '<div id="theme-errors">' . PHP_EOL;
			}
			$results = paramedic_output_results( $themeerrors, $themetest, 'theme' );
			if ( !$paramedic['silent'] ) {
				echo '</div><!-- /#theme-errors -->' . PHP_EOL;
			}
			if ( $results != '' ) {
				paramedic_log_results( $results );
			}
		} elseif ( !$themetest ) {
			echo '<div style="font-size:18px; font-family: Consolas, \'Lucida Console\', Monaco, FreeMono, monospace">';
			echo '<b>' . esc_html( __( 'No matching Themes found to test.', 'paramedic' ) ) . '</b></div><br><br>' . PHP_EOL;
		}
	}

	// maybe output help
	// -----------------
	// if ($_REQUEST['paramedic'] == 'help') {
	//	paramedic_interface();
	//	paramedic_help();
	// }

	// maybe cleanup test logs
	// -----------------------
	if ( isset( $cleanup ) && $cleanup ) {
 		// TODO: do test log cleanups ?
 		// if ($cleanup == 'all') {}
 		// else {}
	}

	// --- maybe display interface menu ---
	if ( !$paramedic['silent'] ) {

		// --- if interface is not already output ---
		if ( !$paramedic['interface'] ) {
			if ( !defined( 'PARAMEDIC_MUSTAUTH' ) ) {
				paramedic_interface();
			} elseif ( !PARAMEDIC_MUSTAUTH ) {
				paramedic_interface();
			} elseif ( PARAMEDIC_MUSTAUTH ) {
				paramedic_recheck_user_auth();
				if ( $paramedic['authed'] ) {
					paramedic_interface();
				}
			}
		}

		// --- close page and exit ---
		if ( $paramedic['interface'] ) {
			echo '</body></html>';
			exit;
		}
	}
}
